%% Laplacian pyramid
%
% Inputs:
%   I = image NxN matrix
%   'N_lev', number of levels in the pyramid (optional)
%   filter = low pass filter.
%
% More information:
%   'The Laplacian Pyramid as a Compact Image Code'
%   Burt, P., and Adelson, E. H., 
%   IEEE Transactions on Communication, COM-31:532-540 (1983). 
%


function [pyr G filter] = laplacian_pyramid_s(Im,N_lev,filter) 

 
if ~exist('N_lev') 
    % compute the highest possible pyramid     
    N_lev = floor(log2(min(size(Im)))); 
end 
 
if ~exist('filter','var')
    f = [.05, .25, .4, .25, .05];  % original [Burt and Adelson, 1983]
    filter = f'*f;
end

% recursively build pyramid 
pyr = cell(N_lev,1); 

J = Im; 
for l = 1:N_lev - 1 
    % apply low pass filter, and downsample 
    G{l} = downsample_s(J,filter); 
    odd = 2*size(G{l}) - size(J);  % for each dimension, check if the upsampled version has to be odd 
    % in each level, store difference between image and upsampled low pass version 
    pyr{l} = J - upsample_s(G{l},odd,filter); 
    J = G{l}; % continue with low pass image 
end 
pyr{N_lev} = J; % the coarest level contains the residual low pass image 


function Im_out = downsample_s(Im, filter) 
 
% low pass, convolve with filter 
Im_out = imfilter(Im,filter,'symmetric');   
 
% decimate 
Im_out = Im_out(1:2:size(Im,1), 1:2:size(Im,2), :);  
 
function Im_out = upsample_s(Im,odd,filter) 
 
% increase resolution 
Im = padarray(Im,[1 1 0],'replicate'); % pad the image with a 1-pixel border 
r = 2*size(Im,1); 
c = 2*size(Im,2); 
k = size(Im,3); 
Im_out = zeros(r,c,k); 
if strcmp(class(Im),'gpuArray') % to be used in a GPU
    Im_out = gpuArray(Im_out);
end
Im_out(1:2:r, 1:2:c, :) = 4*Im; % increase size 2 times; the padding is now 2 pixels wide 
 
% interpolate, convolve with filter 
Im_out = imfilter(Im_out,filter);     
 
% remove the border 
Im_out = Im_out(3:r - 2 - odd(1), 3:c - 2 - odd(2), :);

