
%
% This function performs the distance in the Normalized
% Laplacian Pyramid domain (NLP) for two images.
%
% Moreover (if asked) it computes gradient of the distance:
% dfX = d(NLPdist_lum(X,X'))/d(X)
% Where X is the evaluation point (image) and X' is the original image.
% Note that in order to perform the gradient we employ a convolutional
% version of the Laplacian Pyramid. This is not totally correct and
% therefore the gradient is not perfect (but is a good approximantion).
%
% USE: [fX dfX Im_Xp params_NLP FF] = NLPdist_lum(Im_X,Im_Xp,params_NLP)
% 
% INPUTS:
% Im_X = Evaluation point (image). NxN matrix
%       If Im_X is a gpuArray matrix the function uses the GPU. 
% Im_Xp = Original image. Two options can be given:
%       1) An image, NxN matrix.
%       2) The transformed image in the NLP domain. 
%       This is convenient in a optimization procedure.
%
% INPUTS (optional):
% params_NLP = Parameters of the transformation.
%
% OUTPUTS:
% fX = perceptual distance between Im_X and Im_Xp.
% dfX = Gradient of the function. It is given in vector form. 
%        For matrix form use: Im_dfX = reshape(dfX,size(Im));
% Im_Xp_NLP = Im_Xp in the NLP domain.
% 
%
% NLP distance has shown to be correlated with human perception. 
% Its gradient aims to use NLP distance 
% in image processing problems (instead of the usual MSE).
% The distance was presented in:
% "Perceptually Optimized Rendering"
% V. Laparra, J. Ballé, A. Berardino and E.P. Simoncelli. (2017)
%
% An early statistically fitted version of this distance was presented at:
% "Perceptual image quality assessment using a normalized Laplacian pyramid." 
% V. Laparra, J. Ballé, A. Berardino and E.P. Simoncelli. 
% HVEI: SPIE, Conf. on Human Vision and Electronic Imaging, XXI, 2016, (2016)
%

function [fX dfX Im_Xp_NLP params_NLP AUX] = NLPdist_lum(Im_X,Im_Xp,params_NLP)

if size(Im_X,2) == 1
    Im_X = reshape(Im_X,[sqrt(length(Im_X)) sqrt(length(Im_X))]);
    Im_Xp = reshape(Im_Xp,[sqrt(length(Im_Xp)) sqrt(length(Im_Xp))]);
end

%% INPUTS
if nargout>1
    load('Filters_Laplacian_Pyramid_as_steerable','L_0','H_11','H_12','H_21','H_22')
end
if ~exist('params_NLP','var')
    load(['PAR_NLP_2017'])
    params_NLP = params;
end
if ~isfield(params_NLP,'N_lev')
    N_lev = floor(log2(min(size(Im_X))))-2;
else
    N_lev = params_NLP.N_lev;
end
try(existsOnGPU(Im_X));
    GPU_flag = 1;
catch
    GPU_flag = 0;
end
if iscell(Im_Xp)
    Im_Xp_NLP = Im_Xp;
end

for ii=1:N_lev-1
    DN_filts_aux(ii).F2 = params_NLP.DN_filts(1).F2;
end
DN_filts_aux(N_lev).F2 = params_NLP.DN_filts(end).F2;
params_NLP.DN_filts = DN_filts_aux;

sigmas(1:N_lev-1) = params_NLP.sigmas(1);
sigmas(N_lev) = params_NLP.sigmas(end);

% Transforming Im_Xp (if needed)
if ~iscell(Im_Xp)
    Im_Xpg = Im_Xp.^(1./params_NLP.exp_g);
    Lap_dom_ori = laplacian_pyramid_s(Im_Xpg,N_lev,params_NLP.F1);
    for N_b = 1:N_lev
        LL = floor(size(params_NLP.DN_filts(N_b).F2)/2);
        A2 = padarray(abs(Lap_dom_ori{N_b}),LL,'symmetric');
        DEN = sigmas(N_b) + conv2(A2,params_NLP.DN_filts(N_b).F2,'valid');
        Im_Xp_NLP{N_b} = Lap_dom_ori{N_b} ./ DEN;
        DEN_Xp{N_b} = conv2(A2,params_NLP.DN_filts(N_b).F2,'valid');
    end
end


%% FUNCTION
Im_Xg = Im_X.^(1./params_NLP.exp_g);
Lap_dom = laplacian_pyramid_s(Im_Xg,N_lev,params_NLP.F1);

for N_b = 1:N_lev
    
    LL = floor(size(params_NLP.DN_filts(N_b).F2)/2);
    A2 = padarray(abs(Lap_dom{N_b}),LL,'symmetric');
    DEN = sigmas(N_b) + conv2(A2,params_NLP.DN_filts(N_b).F2,'valid');
  
    Im_X_NLP{N_b} = Lap_dom{N_b} ./ DEN;
    DEN_X{N_b} = conv2(A2,params_NLP.DN_filts(N_b).F2,'valid');
    
    dif{N_b} = Im_X_NLP{N_b}-Im_Xp_NLP{N_b};
    
    fX_aux(N_b) = (mean(abs(dif{N_b}(:)).^params_NLP.exp_s)).^(1/params_NLP.exp_s);
    
    if nargout>1 %(Note that the gradient is computed only if requested)

        % gradient
        AAA = sign(dif{N_b}).*abs(dif{N_b}).^(params_NLP.exp_s-1)./(DEN.^2);
        aux1 = Lap_dom{N_b}.*AAA; 
        
        P = rot90(params_NLP.DN_filts(N_b).F2,2);
        auxb = conv2(aux1,P,'same');
        d_y_z2 = sign(Lap_dom{N_b}).*auxb;
        
        d_y_z1 = DEN.*AAA; 
        
        d_y_z = single(d_y_z1 - d_y_z2);
  
        
        % Laplacian Pyramid derivative
        if N_b<N_lev
            dif_11 = zeros(size(d_y_z));
            if GPU_flag == 1, dif_11 = gpuArray(dif_11); end
            dif_12 = dif_11;
            dif_21 = dif_11;
            dif_22 = dif_11;
            
            dif_11(1:2:end,1:2:end) = d_y_z(1:2:end,1:2:end);
            der_11 = imfilter(dif_11,H_11,'symmetric');
            
            dif_12(1:2:end,2:2:end) = d_y_z(1:2:end,2:2:end);
            der_12 = imfilter(dif_12,H_12,'symmetric');
            
            dif_21(2:2:end,1:2:end) = d_y_z(2:2:end,1:2:end);
            der_21 = imfilter(dif_21,H_21,'symmetric');
            
            dif_22(2:2:end,2:2:end) = d_y_z(2:2:end,2:2:end);
            der_22 = imfilter(dif_22,H_22,'symmetric');
            
            dif_s = der_11 + der_12 + der_21 + der_22;
        else
            dif_s = d_y_z;
        end
        
        for Nb_2 = N_b-1:-1:1
            d_aux = zeros(size(dif{Nb_2}));
            if GPU_flag == 1, d_aux = gpuArray(d_aux); end
            d_aux(1:2:end,1:2:end) = dif_s;
            dif_s = imfilter(d_aux,L_0,'symmetric');
        end
        d_di_xpj = dif_s;

        % Constant per band
        aaa = real((sum(abs(dif{N_b}(:)).^params_NLP.exp_s))^(params_NLP.exp_f/params_NLP.exp_s-1));
        if fX_aux(N_b) == 0, aaa = 0; end
        Nc = length(dif{N_b}(:));
        konst = aaa * (1/(Nc.^(params_NLP.exp_f/params_NLP.exp_s)));
        
        dfX_aux2b(:,N_b) = konst*(d_di_xpj(:).*((1./params_NLP.exp_g)*Im_X(:).^((1./params_NLP.exp_g)-1)))';
    end
end


fX = mean(fX_aux.^params_NLP.exp_f).^(1/params_NLP.exp_f);

if nargout>1 %(Note that the gradient is computed only if requested)
    if fX == 0
        dfX = zeros(size(dfX_aux2b,1),1);
    else
        dfX = fX^(1-params_NLP.exp_f)*mean(dfX_aux2b,2);
    end
end

if nargout>4
    AUX.fX_freq = fX_aux;
    AUX.map = dif;
    AUX.Im_X_NLP = Im_X_NLP;
    AUX.Im_Xp_NLP = Im_Xp_NLP;
    AUX.Im_X_LP = Lap_dom;
    AUX.Im_Xp_LP = Lap_dom_ori;
    AUX.DEN_X = DEN_X;
    AUX.DEN_Xp = DEN_Xp;
end


