
%
% USE:
% Im_ldr = NLP_rendering(Im,params)
%
% INPUTS: 
% - Im = Original image in grayscale and cd/m2.
%
%   Note that if the image is in digital counts i.e. Im_dg is in [0-1] range, 
%   it can be easily approximatelly converted to cd/m2 by, unduing the
%   gamma correction and puting it in the screen luminance range:
%   Im = (Im_dg.^gc)*(LM_out-Lm_out)+Lm_out;
%
% - params_scr (optional) = struct with the parameters of the screen
% default:
% params_scr.LM_out = 300; % assuming images in screen have maximum 140 cd/m²
% params_scr.Lm_out = 5; % assuming images in screen have minimum 10 cd/m²
% params_scr.gc = 2.2; % gamma correction factor
%
% - params_NLP (optional) = struct with the parameters of the NLP (see
% NLPdist_lum.m for details)
%
% OUTPUTS:
%
% Im_out = Ouput image in cd/m2
% fX_n = evolution of the objective during the optimization
% Im_out_aux = Output image prepared to be shown in the screen, range [0 1].
%
%
% Optimization procedure using the NLP transform presented in the paper:
% "Perceptually Optimized Imgae Rendering"
% V. Laparra, A. Berardino, J. Balle & E.P. Simoncelli (2017)
%


function [Im_out fX_n Im_out_scr] = NLP_rendering(Im,params_scr,params_NLP)

%% PARAMETERS

if ~exist('params_scr','var')
    LM_out = 300; % asuming images in screen have maximum 140 cd/m²
    Lm_out = 5; % asuming images in screen have minimum 10 cd/m²
    gc = 2.2; % gamma correction factor
    N_steps = 1000;
else
    if isfield(params_scr,'LM_out')
        LM_out = params_scr.LM_out;
    else
        LM_out = 140; % asuming images in screen have maximum 140 cd/m²
    end
    if isfield(params_scr,'Lm_out')
        Lm_out = params_scr.Lm_out;
    else
        Lm_out = 10; % asuming images in screen have maximum 140 cd/m²
    end
    if isfield(params_scr,'gc')
        gc = params_scr.gc;
    else
        gc = 2.2; % asuming images in screen have maximum 140 cd/m²
    end
    if isfield(params_scr,'N_steps')
        N_steps = params_scr.N_steps;
    else
        N_steps = 1000; % maximum number of steps for the optimization procedure
    end
end
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% NLP OPTIMIZATION (Using ADAM gradient optimization)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Original Image in NLP domain and parameters
if ~exist('params_NLP','var')
    [~,~,Im_DN,params_NLP] = NLPdist_lum(Im,Im);
else
    [~,~,Im_DN,params_NLP] = NLPdist_lum(Im,Im,params_NLP); 
end

% Initializing with a rescaled version
if isfield(params_scr,'ImR')
    ImR = params_scr.ImR;
else
    Im_g = Im.^(1/2.2);
    %Im_g = Im;
    aux = (Im_g-min(Im_g(:)))/range(Im_g(:));
    ImR = Lm_out + aux*(LM_out-Lm_out);
end
% GD params

Im_n = ImR(:);
fX_n_post = []; fX_n = [];

% ADAM params
m = 0; v = 0; beta_1 = 0.9; beta_2 = 0.999; ep = 10e-8;

% Set alpha parameter automatically
if isfield(params_scr,'alpha')
    alpha = params_scr.alpha;
else
    alpha = 2*max(Im_n);
    [fX_n dfX] = NLPdist_lum(reshape(Im_n,size(ImR)),Im_DN,params_NLP);
    m_h = (1-beta_1)*dfX/(1-beta_1);
    v_h = (1-beta_2)*dfX.^2/(1-beta_2);
    fX_n_2 = fX_n + 1;
    while fX_n_2>fX_n
        alpha = alpha/2;
        Im_n_aux = Im_n - alpha.*m_h./(sqrt(v_h) + ep);
        
        Im_n_aux = min(Im_n_aux,LM_out);
        Im_n_aux = max(Im_n_aux,Lm_out);
        
        fX_n_2 = NLPdist_lum(reshape(Im_n_aux,size(ImR)),Im_DN,params_NLP);
    end
    alpha = alpha/10; % something conservative
end

% Gradient descent
for nn = 1:N_steps
    tic
    % Gradient
    [fX_n(nn) dfX] = NLPdist_lum(reshape(Im_n,size(ImR)),Im_DN,params_NLP);
    
    % ADAM step
    m = beta_1*m + (1-beta_1)*dfX;
    v = beta_2*v + (1-beta_2)*dfX.^2;
    m_h = m/(1-beta_1.^nn);
    v_h = v/(1-beta_2.^nn);
    
    Im_n = Im_n - alpha.*m_h./(sqrt(v_h) + ep);
    
    % Constrains projection
    Im_n = min(Im_n,LM_out);
    Im_n = max(Im_n,Lm_out);
    
    % STOP criteria
    if nn>1
        if fX_n(nn)>=fX_n(nn-1)-10e-7, alpha = alpha/2; end
        if alpha<1e-5, break, end
    end
    % show evolution
    [nn/N_steps fX_n(nn)/fX_n(1) alpha toc]
end

% Extimated image
Im_out = reshape(Im_n,size(Im));
Im_out_scr = ((Im_out-Lm_out)/(LM_out-Lm_out)).^(1/gc);


