function im = textureSynth(params, im0, Niter, imask, pmask)

% Synthesize texture applying Portilla-Simoncelli model/algorithm,
% using AdjustCorr2 to impose the pointwise cross-correlation through
% the different orientations and scales (coarse to fine approach).
% Current result and parameter SNRs are shown in two new figures.
%
% [res] = textureSynth(params, initialIm, Niter, imask, pmask)
%	
% params: structure containing texture parameters (as returned by textureAnlz).
%
% initialIm: initial image, OR a 2-vector containing dimensions of desired
% image (if dimensions are passed, initial image is Gaussian white
% noise).
%
% Niter (optional): Number of iterations.  Default = 50.
%
% imask (optional): imsizex2 matrix.  First column is a binary mask, second
% column contains the image values to be imposed.
%
% pmask (optional): pyrsizex2 matrix.  First column is a binary mask, second
% column contains the (real) pyramid coefficients to be imposed.

% 7/25/99 (EPS): Adapted from the sintexcEEE.m function.

%% Check required args are passed:
if (nargin < 2)
  error('Function called with too few input arguments');
end

%% *** Would be nice to allow an option to run until convergence!
if ( exist('Niter') ~=  1 | isempty(Niter) )
  Niter = 50;
end

if (( exist('imask') == 1 ) & ~isempty(imask) )
  imaskInd = find( imask(:,1) > 0.5 );  % indices of ones in mask
else
  imaskInd = [];
end

if (( exist('pmask') == 1 ) & ~isempty(pmask) )
  pmaskInd = find( pmask(:,1) > 0.5 );  % indices of ones in mask
else
  pmaskInd = [];
end

%% 1D interpolation filter, for scale cross-correlations:
interp = [-1/16 0 9/16 1 9/16 0 -1/16];

%% Extract parameters:  (*** this should be cleaned up)
statg0 = params.pixelStats;
mean0 = statg0(1); var0 =  statg0(2);
skew0 =  statg0(3); kurt0 =  statg0(4);
mn0 =  statg0(5);  mx0 = statg0(6);
%statl0 = params.lowStats;
%stath0 = params.highStats;
acr0 = params.autoCorrReal;
ace0 = params.autoCorrMag;
magMeans0 = params.magMeans;
C0 = params.cousinMagCorr;
Cx0 = params.parentMagCorr;

%% Extract {Nsc, Nor, Na} from params
tmp = size(params.autoCorrReal);
Na = tmp(1); Nsc = tmp(3); Nor = tmp(4);
la = (Na-1)/2;

%% If im is a vector of length 2, create Gaussian white noise image of this
%% size, with desired pixel mean and variance.
if ( length(im0) == 2 )
  im = mean0 + sqrt(var0)*randn(im0);
else
  im = im0;
end

%% If the spatial neighborhood Na is too big for the lower scales,
%% "modacor22.m" will make it as big as the spatial support at
%% each scale:
%% *** CHECK THIS:
[Ny,Nx] = size(im);
nth = log2(min(Ny,Nx)/Na);
if nth<Nsc+1,
  fprintf(1,'Warning: Na will be cut off for levels above #%d !\n',floor(nth));
end

if ( ~isempty(imaskInd) &  (size(imask,1) ~= prod(size(im0))) )
  error(sprintf('imask size %d does not match image dimensions [%d,%d]',...
		size(imask,1), size(im0,1), size(im0,2)));
end

imf = max(1,gcf-1); snrf = imf+1;
figure(imf);  clf
subplot(1,2,1); grayRange = showIm(im); title('Starting image');
drawnow

%% MAIN LOOP
for niter = 1:Niter

  %% Build the steerable pyramid
  [pyr,pind] = buildSCFpyr(im,Nsc,Nor-1);
  apyr = abs(pyr);

  if ( any(vectorize(mod(pind,4))) )
    error('Algorithm will fail: band dimensions are not all multiples of 4!');
  end

  if ( ~isempty(pmaskInd) &  (size(pmask,1) ~= size(pyr,1)) )
    error(sprintf('pmask size %d does not match pyramid size %d',...
		  size(pmask,1), size(pyr,1)));
  end

  %% Subtract mean of magnitude
  magMeans = zeros(size(pind,1), 1);
  for nband = 1:size(pind,1)
    indices = pyrBandIndices(pind,nband);
    magMeans(nband) = mean2(apyr(indices));
    apyr(indices) = apyr(indices) - magMeans(nband);
  end

  %% Adjust autoCorr of lowBand
  nband = size(pind,1); 
  ch = pyrBand(apyr,pind,nband);
  Sch = min(size(ch)/2);
  nz = sum(sum(~isnan(ace0(:,:,Nsc+1,1))));
  lz = (sqrt(nz)-1)/2;
  le = min(Sch/2-1,lz);
  ch = modacor22(ch, ace0(la-le+1:la+le+1,la-le+1:la+le+1,Nsc+1,1));
  if (var2(imag(ch))/var2(real(ch)) > 1e-6)
    fprintf(1,'Discarding non-trivial imaginary part!');
  end
  ch = real(ch);
  apyr(pyrBandIndices(pind,nband)) = ch;

  %% Coarse-to-fine loop:
  for nsc = Nsc:-1:1
    
    firstBnum = (nsc-1)*Nor+2;
    cousinSz = prod(pind(firstBnum,:));
    ind = pyrBandIndices(pind,firstBnum);
    cousinInd = ind(1) + [0:Nor*cousinSz-1];

    %% Interpolate parents
    if (nsc<Nsc)
      parents = zeros(cousinSz,Nor);
      for nor = 1:Nor
	nband = (nsc+1-1)*Nor+nor+1; 
	tmp = pyrBand(apyr, pind, nband);
	tmp = upConv(tmp, interp, 'reflect1', [1 2], [1 1], [size(tmp,1) pind(firstBnum,2)]);
	tmp = upConv(tmp, interp', 'reflect1', [2 1], [1 1], pind(firstBnum,:));
	parents(:,nor) = vectorize(tmp);
      end
      nparents =  Nor;
    else
      tmp = pyrLow(apyr,pind);
      tmp = upConv(tmp, interp, 'reflect1', [1 2], [1 1], [size(tmp,1)  pind(firstBnum,2)]);
      tmp = upConv(tmp, interp', 'reflect1', [2 1], [1 1], pind(firstBnum,:));
      parents = vectorize(tmp);
      nparents = 1;
    end
 
    %% Adjust autoCorr of energy 
    nband = (nsc-1)*Nor+2; 
    Sch = min(pind(nband,:)/2);
    nz = sum(sum(~isnan(acr0(:,:,nsc,1))));
    lz = (sqrt(nz)-1)/2;
    le = min(Sch/2-1,lz);
    for nor = 1:Nor,
      nband = (nsc-1)*Nor+nor+1; 
      ch = pyrBand(apyr,pind,nband);
      [ch, snr1(niter,nband-1)] = modacor22(ch,...
	ace0(la-le+1:la+le+1,la-le+1:la+le+1,nsc,nor));
      ch = real(ch);
      apyr(pyrBandIndices(pind,nband)) = ch;
    end
    
    %% Adjust cross-correlation with other orientations and parents:
    cousins = reshape(apyr(cousinInd), [cousinSz Nor]);
    [cousins, snr3(niter,nsc), snr4(niter,nsc)] = ...
	adjustCorr2s(cousins, C0(:,:,nsc), parents, Cx0(:,[1:nparents],nsc), 2);
    if (var2(imag(cousins))/var2(real(cousins)) > 1e-6)
      fprintf(1,'Discarding non-trivial imaginary part!');
    end
    cousins = real(cousins);
    apyr(cousinInd) = vectorize(cousins);

    %% Adjust the raw autoCorr, add mean back to magnitudes, and combine
    for nor = 1:Nor,
      nband = (nsc-1)*Nor+nor+1; 
      ind = pyrBandIndices(pind,nband);
      ch = pyrBand(pyr, pind, nband);
      [ch, snr2(niter,nband-1)] = modacor22(ch,...
	acr0(la-le+1:la+le+1,la-le+1:la+le+1,nsc,nor));
      pyr(ind) = ch;

      mag = apyr(ind) + magMeans0(nband);
      mag = mag .* (mag>0);
      pyr(ind) = pyr(ind) .* (mag./abs(pyr(ind)));
      mag = abs(pyr(ind));
      apyr(ind) = mag - mean2(mag); 
    end

  end  %END Coarse-to-fine loop

  %% Adjust stats of lowBand 
  ind = pyrBandIndices(pind,Nsc*Nor+2);
  apyr(ind) = apyr(ind) + magMeans0(Nsc*Nor+2);
  ch = pyr(ind).* apyr(ind)./abs(pyr(ind));
%  [mn mx] = range2(ch);
%  statl = [mean2(ch) var2(ch) mn mx]; 
%  ch = (ch-statl(1)) * sqrt(statl0(2)/statl(2)) + statl0(1);
%  ch = max(min(ch,statl0(4)),statl0(3));
  pyr(ind) = ch;

  if (0) %% COMMENTED
     %% Adjust stats of HP
     ind = pyrBandIndices(pind,1);
     ch = pyr(ind);
     [mn mx]=range2(ch);
     stath = [mean2(ch) var2(ch) mn mx]; 
     ch = (ch-stath(1)) * sqrt(stath0(2)/stath(2)) + stath0(1);
     ch = max(min(ch,stath0(4)), stath0(3));
     pyr(ind) = ch;
  end %% COMMENT

  rpyr = real(pyr);

  %% Force coefficients specified by pyramid mask
  if (~isempty(pmaskInd))
    rpyr(pmaskInd) = pmask(pmaskInd, 2);
  end

  %% Reconstruct an image from the real part of the pyramid
  im = reconSFpyr(rpyr,pind);

  %% Pixel statistics
  [mns mxs] = range2(im);
  means = mean2(im);
  vars = var2(im, means);
  skews = skew2(im, means, vars);
  kurts = kurt2(im, means, vars);
  snr5(niter,1) = snr(mean0,mean0-means);
  snr5(niter,2) = snr(var0,var0-vars);
  snr5(niter,3) = snr(skew0,skew0-skews);
  snr5(niter,4) = snr(kurt0,kurt0-kurts);
  snr5(niter,5) = snr(mx0-mn0,sqrt((mx0-mxs)^2+(mn0-mns)^2));  
  %% min could be 0, so have to do something unusual here....

  im = im-means;  			% Adjusts mean and variance
  im = im*sqrt(var0/var2(im));	
  im = im+mean0;
  im = modskew(im,skew0);		% Adjusts skewness (keep mean and variance)
  im = modkurt(im,kurt0);		% Adjusts kurtosis (keep mean and variance,
					% but not skewness)
  im = max(min(im,mx0),mn0);		% Adjusts range (affects everything)	

  %% Force pixels specified by image mask
  if (~isempty(imaskInd))
    im(imaskInd) = imask(imaskInd, 2);
  end
  
  figure(imf); subplot(1,2,2);
  showIm(im); title(sprintf('iteration %d',niter)); 

  figure(snrf);
  subplot(151); plot(snr1); title('Mag autoCorr');
  subplot(152); plot(snr2); title('Raw autoCorr');
  subplot(153); plot(snr3); title('Ori crossCorr');
  subplot(154); plot(snr4); title('Scale crossCorr');
  subplot(155); plot(snr5); title('Pixel Stats');
  drawnow

end %END  MAIN LOOP
