function [params] = textureAnlz(im0, Nsc, Nor, Na)

% Analyze texture for application of Portilla-Simoncelli model/algorithm.
%
% [params] = textureAnlz(im0, Nsc, Nor, Na);
% 	im0: 	original image
% 	Nsc: 	number of scales
% 	Nor: 	number of orientations
% 	Na:	spatial neighborhood considered (Na x Na)	
%
% Example: Nsc=4; Nor=4; Na=7;
%
% See also textureSynth.

% 7/25/99 (EPS): Adapted from the sintexcEEE.m function.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Check required args are passed
if (nargin < 4)
  error('Function called with too few input arguments');
end

%% 1D interpolation filter, for scale cross-correlations:
interp = [-1/16 0 9/16 1 9/16 0 -1/16];

if ( mod(Na,2) == 0 )
  error('Na is not an odd integer');
end

%% If the spatial neighborhood Na is too big for the lower scales,
%% "modacor22.m" will make it as big as the spatial support at
%% each scale:

[Ny,Nx] = size(im0);
nth = log2(min(Ny,Nx)/Na);
if nth<Nsc,
  fprintf(1,'Warning: Na will be cut off for levels above #%d !\n', floor(nth+1));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

la=floor((Na-1)/2);

%% Pixel statistics
[mn0 mx0] = range2(im0);
mean0 = mean2(im0);
var0 = var2(im0, mean0);
skew0 = skew2(im0, mean0, var0);
kurt0 = kurt2(im0, mean0, var0);
statg0 = [mean0 var0 skew0 kurt0 mn0 mx0];

%% Build the steerable pyramid
[pyr0,pind0] = buildSCFpyr(im0,Nsc,Nor-1);
rpyr0 = real(pyr0);
apyr0 = abs(pyr0);

clear pyr0

if ( any(vectorize(mod(pind0,2))) )
  error('Algorithm will fail: Some bands have odd dimensions!');
end

figure(gcf)
showIm(im0); title('Original'); 

figure(gcf+1)
showSpyr(rpyr0,pind0);

%% Subtract mean of magnitude:
magMeans0 = zeros(size(pind0,1), 1);
for nband = 1:size(pind0,1)
  indices = pyrBandIndices(pind0,nband);
  magMeans0(nband) = mean2(apyr0(indices));
  apyr0(indices) = apyr0(indices) - magMeans0(nband);
end

%% Compute  central autoCorr of the channels (real part and energy).
ace = NaN * ones(Na,Na,Nsc+1,Nor);
acr = NaN * ones(Na,Na,Nsc,Nor);
for nsc = 1:Nsc,
  for nor = 1:Nor,
    nband = (nsc-1)*Nor+nor+1;
    ch = pyrBand(apyr0,pind0,nband);
    [Nlx Nly] = size(ch);
    cx = Nlx/2+1;  %Assumes Nlx even
    cy = Nly/2+1;
    Sch = min(Nlx, Nly);
    le = min(Sch/2-1,la);
    ac = fftshift(real(ifft2(abs(fft2(ch)).^2)))/prod(size(ch));
    ac = ac(cy-le:cy+le,cx-le:cx+le);
    ace(la-le+1:la+le+1,la-le+1:la+le+1,nsc,nor) = ac;
    ch = pyrBand(rpyr0,pind0,nband);
    ac = fftshift(real(ifft2(abs(fft2(ch)).^2)))/prod(size(ch));
    ac = ac(cy-le:cy+le,cx-le:cx+le);
    acr(la-le+1:la+le+1,la-le+1:la+le+1,nsc,nor) = ac;
  end
end

%% Compute central autoCorr of lowband  
ch = pyrLow(apyr0,pind0);
[Nly Nlx] = size(ch);
Sch = min(Nly,Nlx); %size of low band
le = min(Sch/2-1,la);
cy = Nly/2+1;
cx = Nlx/2+1;
ac = fftshift(real(ifft2(abs(fft2(ch)).^2)))/prod(size(ch));
ac = ac(cy-le:cy+le,cx-le:cx+le);
ace(la-le+1:la+le+1,la-le+1:la+le+1,Nsc+1,1) = ac;

%% Compute the cross-correlation matrices of the original
%% pyramid at the different levels 

C0 = zeros(Nor,Nor,Nsc);
Cx0 = zeros(Nor,Nor,Nsc);

for nsc = 1:Nsc,

  firstBnum = (nsc-1)*Nor+2;
  cousinSz = prod(pind0(firstBnum,:));
  ind = pyrBandIndices(pind0,firstBnum);
  cousinInd = ind(1) + [0:Nor*cousinSz-1];

  if (nsc<Nsc)
    parents = zeros(cousinSz,Nor);
    for nor=1:Nor,
      nband = (nsc-1+1)*Nor+nor+1;
      tmp = pyrBand(apyr0, pind0, nband);
      tmp = upConv(tmp, interp, 'reflect1', [1 2], [1 1], [size(tmp,1) pind0(firstBnum,2)]);
      tmp = upConv(tmp, interp', 'reflect1', [2 1], [1 1], pind0(firstBnum,:));
      parents(:,nor) = vectorize(tmp);
    end
    nparents = Nor;
  else
    tmp = pyrLow(apyr0,pind0);
    tmp = upConv(tmp, interp, 'reflect1', [1 2], [1 1], [size(tmp,1)  pind0(firstBnum,2)]);
    tmp = upConv(tmp, interp', 'reflect1', [2 1], [1 1], pind0(firstBnum,:));
    parents = vectorize(tmp);
    nparents = 1;
  end
  cousins = reshape(apyr0(cousinInd), [cousinSz Nor]);
  C0(:,:,nsc) = innerProd(cousins)/cousinSz;
  Cx0(:,[1:nparents],nsc) = (cousins'*parents)/cousinSz;
end

%% Calculate the mean, range and variance of the LF and HF residuals' energy.
%channel = rpyr0(pyrBandIndices(pind0,Nsc*Nor+2));
%[mn mx] = range2(channel);
%statl0 = [mean2(channel) var2(channel) mn mx];

%channel = apyr0(pyrBandIndices(pind0,1));
%[mn mx] = range2(channel);
%stath0 = [mean2(channel) var2(channel) mn mx]; 

params = struct('pixelStats', statg0, ...
                'autoCorrReal', acr, ...
                'autoCorrMag', ace, ...
		'magMeans', magMeans0, ...
                'cousinMagCorr', C0, ...
                'parentMagCorr', Cx0);

                % 'lowStats', statl0, ...
                % 'highStats', stath0, ...

