% [Vy, Vx] = match(FRAME1, FRAME2, BLUR, LEVELS, SIGP)
%
% Compute a translational vector field that matches FRAME1 to FRAME2, using a
% coarse-to-fine algorithm based on the standard differential
% brightness constancy constraint.
%
% BLUR (optional) specifies the region over which the constraint
% combination occurs.  It can be either a convolution kernel (vector
% or matrix), a string naming a filter (see namedFilter), 'none', or
% 'all' (combine over whole image).  Default is 'binom3'.
%
% LEVELS specifies the number of pyramid levels 
%    (default = maxPyrHt(size(im1),[5 5]))
%
% SIGP (optional) is the inverse standard deviation of a Gaussian
% prior, which is used at each level in calculating the residual warp.
% (default = 0.5).

%% EPS, spring '96.
%% revised 8/97:

%% Algorithm reference:
%% E P Simoncelli and E H Adelson and D J Heeger.
%% "Probability Distributions of Optical Flow"
%% IEEE Conf on Computer Vision and Pattern Recognition, Mauii, Hawaii.
%% June, 1991.

function [vy, vx] = match(im1, im2, blur, levels, sigp)

%% -----------------------------------------------------------------
%% Parameter processing:

%% Check: size of images should be same
if ( size(im1) ~= size(im2) )
  error('Images must be of equal size!');
else
  [ysz,xsz] = size(im1);
end

if (exist('blur') ~= 1)
  blur = 'binom3';
end

%% If levels argument was not included, compute it automatically:
if (~exist('levels'))
  levels = maxPyrHt(size(im1),[5 5])-1;
end

%%% *** OLD:
%% If sigp was not included, set it to the signal-to-noise ratio of an
%% 8-bit quantized signal.
%%%  Osqrt((1/12) / var(im1(:)));

if (~exist('sigp'))
  sigp = 0.5;
end

%% -----------------------------------------------------------------
%% Displacement Calculation:

if (levels <= 1)  %% top level

  [vy, vx] = diffMatch(im1, im2, blur, sigp);

else

  %% Blur and subsample images:
  gs = namedFilter('binom5')/sqrt(2);
  im1bs = blurDn(im1,1,gs);
  im2bs = blurDn(im2,1,gs);

  %% Match the blurred and subsampled images:
  [vyps, vxps] = match(im1bs, im2bs, levels-1, sigp);
  
  %% Compute predicted flowfields (upsample and interpolate):
  gp = sqrt(2) * namedFilter('binom3');    
  vxp = upBlur(vxps,1,gp);  
  vyp = upBlur(vyps,1,gp);  

  %% warp im1 forward and im2 backward using predicted flowfields vxp and vyp
  im1p = warpIm(im1, vyp, vxp);
  im2p = warpIm(im2, -vyp, -vxp);

  %% compute residual displacement
  [rvy, rvx] = diffMatch(im1p, im2p, blur, sigp);

  %% add  predicted and residual displacements to get new displacement estimates
  vy = rvy + 2 .* vyp;
  vx = rvx + 2 .* vxp;
  
end  

