/* 
[N, Y, X] = jhisto(Yimage, Ximage, NBINS_OR_BINSIZE, BIN_CENTER)
  Compute joint histograms.
  EPS, 4/97.
*/

/* Matlab V4 types should be changed as follows:
  Matrix -> mxArray
  REAL -> mxREAL
  mxCreateFull ->  mxCreateDoubleMatrix
  */

#define V4_COMPAT
#include <matrix.h>  /* Matlab matrices */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <strings.h>

#include <stdlib.h>

#define notDblMtx(it) (!mxIsNumeric(it) || !mxIsDouble(it) || mxIsSparse(it) || mxIsComplex(it))

#define PAD 0.49999 /* A hair below 1/2, to avoid roundoff errors */
#define MAXBINS 4000

void mexFunction(int nlhs,	     /* Num return vals on lhs */
		 Matrix *plhs[],    /* Matrices on lhs      */
		 int nrhs,	     /* Num args on rhs    */
		 Matrix *prhs[]     /* Matrices on rhs */
		 )
  {
  register double temp;
  register int binnum1, binnum2, i, size;
  register double *im1, *im2, binsize1, binsize2;
  register double origin1, origin2, *hist;
  register double mn1, mx1, mean1, mn2, mx2, mean2;
  register int nbins1, nbins2;
  double *bincenters; 
  Matrix *arg;
  double *mxMat;

  if (nrhs < 2 ) mexErrMsgTxt("requires at least 2 arguments.");

  /* ARG 1: YMATRIX  */
  arg = prhs[0];
  if notDblMtx(arg) mexErrMsgTxt("MTX arg must be a real non-sparse matrix.");
  im1 = mxGetPr(arg);
  size = (int) mxGetM(arg) * mxGetN(arg);

  /* ARG 2: XMATRIX  */
  arg = prhs[1];
  if notDblMtx(arg) mexErrMsgTxt("MTX arg must be a real non-sparse matrix.");
  im2 = mxGetPr(arg);
  if ( size != ((int) mxGetM(arg) * mxGetN(arg)))
    mexErrMsgTxt("Matrix args are not the same size.");
  

  /* NOTE: uses binsize as a dummy var */
  mn1 = *im1;   mx1 = *im1;  binsize1 = 0;
  for (i=1; i<size; i++)
    {
      temp = im1[i];
      if (temp < mn1)
	mn1 = temp;
      else if (temp > mx1)
	mx1 = temp;
      binsize1 += temp;
    }
  mean1 = binsize1 / size;

  mn2 = *im2;   mx2 = *im2;  binsize2 = 0;
  for (i=2; i<size; i++)
    {
      temp = im2[i];
      if (temp < mn2)
	mn2 = temp;
      else if (temp > mx2)
	mx2 = temp;
      binsize2 += temp;
    }
  mean2 = binsize2 / size;

  /* ARG 4: BIN_CENTER */
  if (nrhs > 3)
    {
    arg = prhs[3];
    if notDblMtx(arg) mexErrMsgTxt("BIN_CENTER arg must be a real scalar or 2-vector.");
    mxMat= mxGetPr(arg);
    if (mxGetM(arg) * mxGetN(arg) == 1)
	{ origin1 = *mxMat; origin2 =  *mxMat; }
    else if (mxGetM(arg) * mxGetN(arg) == 2)
	{ origin1 = mxMat[0]; origin2 = mxMat[1]; }
    else mexErrMsgTxt("BIN_CENTER must be a real scalar, or 2-vector.");
    }
  else
    { origin1 = mean1; origin2 = mean2; }

  /* ARG 3: If positive, NBINS.  If negative, -BINSIZE. */
  if (nrhs > 2)
    {
    arg = prhs[2];
    if notDblMtx(arg) mexErrMsgTxt("NBINS_OR_BINSIZE arg must be a real scalar or 2-vector.");
    mxMat= mxGetPr(arg);
    if (mxGetM(arg) * mxGetN(arg) == 1)
	{ binsize1 = *mxMat; binsize2 = *mxMat; }
    else if (mxGetM(arg) * mxGetN(arg) == 2)
	{ binsize1 = mxMat[0]; binsize2 = mxMat[1]; }
    else
      mexErrMsgTxt("NBINS_OR_BINSIZE must be a real scalar or 2-vector.");
    }
  else  /* MAGIC NUMBER DEFAULT: 101 bins */
    { binsize1 = 101; binsize2 = 101; }
  
  /* --------------------------------------------------
     Adjust origin, binsize, nbins such that
        mx <= origin + (nbins-1)*binsize + PAD*binsize
	mn >= origin - PAD*binsize
     -------------------------------------------------- */
  if (binsize1 < 0)		/* user specified BINSIZE1 */
      {
      binsize1 = -binsize1;
      origin1 -= binsize1 * ceil((origin1-mn1-PAD*binsize1)/binsize1);
      nbins1 = (int) ceil((mx1-origin1-PAD*binsize1)/binsize1) +1;
      }
  else				/* user specified NBINS1 */
      {
      nbins1 = (int) (binsize1 + 0.5);    /* round to int */
      if (nbins1 == 0)
	mexErrMsgTxt("NBINS1 must be greater than zero.");
      binsize1 = (mx1-mn1)/(nbins1-1+2*PAD);   /* start with lower bound */
      i = ceil((origin1-mn1-binsize1/2)/binsize1);
      if ( mn1 < (origin1-i*binsize1-PAD*binsize1) )
	binsize1 = (origin1-mn1)/(i+PAD);
      else if ( mx1 > (origin1+(nbins1-1-i)*binsize1+PAD*binsize1) )
	binsize1 = (mx1-origin1)/((nbins1-1-i)+PAD);
      origin1 -= binsize1 * ceil((origin1-mn1-PAD*binsize1)/binsize1);
      }

  if (binsize2 < 0)		/* user specified BINSIZE2 */
      {
      binsize2 = -binsize2;
      origin2 -= binsize2 * ceil((origin2-mn2-PAD*binsize2)/binsize2);
      nbins2 = (int) ceil((mx2-origin2-PAD*binsize2)/binsize2) +1;
      }
  else				/* user specified NBINS2 */
      {
      nbins2 = (int) (binsize2 + 0.5);    /* round to int */
      if (nbins2 == 0)
	mexErrMsgTxt("NBINS2 must be greater than zero.");
      binsize2 = (mx2-mn2)/(nbins2-1+2*PAD);   /* start with lower bound */
      i = ceil((origin2-mn2-binsize2/2)/binsize2);
      if ( mn2 < (origin2-i*binsize2-PAD*binsize2) )
	binsize2 = (origin2-mn2)/(i+PAD);
      else if ( mx2 > (origin2+(nbins2-1-i)*binsize2+PAD*binsize2) )
	binsize2 = (mx2-origin2)/((nbins2-1-i)+PAD);
      origin2 -= binsize2 * ceil((origin2-mn2-PAD*binsize2)/binsize2);
      } 

if (nbins1 > MAXBINS)
      {
      mexPrintf("nbins1: %d,  MAXBINS: %d\n",nbins1,MAXBINS);
      mexErrMsgTxt("Number of histo bins has exceeded maximum");
      }
if (nbins2 > MAXBINS)
      {
      mexPrintf("nbins2: %d,  MAXBINS: %d\n",nbins2,MAXBINS);
      mexErrMsgTxt("Number of histo bins has exceeded maximum");
      }

  /* Allocate hist  and xvals */
  plhs[0] = (Matrix *) mxCreateFull(nbins1,nbins2,REAL);
  if (plhs[0] == NULL) mexErrMsgTxt("Error allocating result matrix");
  hist = mxGetPr(plhs[0]);

  if (nlhs > 1)
      {
      plhs[1] = (Matrix *) mxCreateFull(1,nbins1,REAL);
      if (plhs[1] == NULL) mexErrMsgTxt("Error allocating result matrix");
      bincenters = mxGetPr(plhs[1]);
      for (i=0, temp=origin1; i<nbins1; i++, temp+=binsize1)
	bincenters[i] = temp;
      }

  if (nlhs > 2)
      {
      plhs[2] = (Matrix *) mxCreateFull(1,nbins2,REAL);
      if (plhs[2] == NULL) mexErrMsgTxt("Error allocating result matrix");
      bincenters = mxGetPr(plhs[2]);
      for (i=0, temp=origin2; i<nbins2; i++, temp+=binsize2)
	bincenters[i] = temp;
      }

  for (i=0; i<size; i++)
      {
      binnum1 = (int) ((im1[i] - origin1)/binsize1 + 0.5);
      binnum2 = (int) ((im2[i] - origin2)/binsize2 + 0.5);
      if ((binnum1 < nbins1) && (binnum1 >= 0))
	  {
	  if ((binnum2 < nbins2) && (binnum2 >= 0))
	    (hist[nbins1*binnum2+binnum1]) += 1.0;
	  else
	    printf("HISTO warning: X image value %f outside of range [%f,%f]\n",
		   im2[i], origin2-0.5*binsize2, origin2+(nbins2-0.5)*binsize2);
	  }
      else
	printf("HISTO warning: Y image value %f outside of range [%f,%f]\n",
	       im1[i], origin1-0.5*binsize1, origin1+(nbins1-0.5)*binsize1);
      }

  return;
  }      

