% [Vy, Vx] = diffMatch(FRAME1, FRAME2, BLUR, SIGP)
%
% Compute a displacement field between FRAME1 and FRAME2,
% using the standard differential brightness constancy constraint.
% This only works for displacements smaller than 2 pixels.  Call
% "match" for larger motions.
%
% BLUR (optional) should be either a convolution kernel (vector or matrix), a
% string naming a filter (see namedFilter), 'none', or 'all' (average
% over whole image).  Default is 'binom3'.
%
% SIGP (optional) is the inverse standard deviation of a Gaussian prior.

%% EPS, Spring '96

function [vy, vx] = diffMatch(frame1, frame2, blur, sigp)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Optional arguments:

if (exist('blur') ~= 1)
  blur = 'binom3';
end

% If BLUR is a string not in ['all', 'none'], assume it's a filter
% name and look up the filter.
if (ischar(blur) & ~strcmp(blur,'all') & ~strcmp(blur,'none'))
  blur = namedFilter(blur);
end

%% If sigp was not included, set it to the signal-to-noise ratio of the
%% 8-bit quantized signal.
if (~exist('sigp'))
  sigp = sqrt((1/12) / var(frame1(:)));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% derivative filters

%% Derivative kernels

%  Design of Multi-dimensional Derivative Filters.  E Simoncelli.
%  IEEE Conf. Image Processing, Austin TX.  1994.
%gp=[0.221830965479  0.556338069043  0.221830965479];
%gd=[-0.461704565162 0.000000000000  0.461704565162];
%gp = [0.035043301565 0.248784801165 0.432343794538 0.248784801165 0.035043301565];
%gd = [-0.106888417209 -0.284612090069 0.000000000000  0.284612090069  0.106888417209];

%% From Farid & Simoncelli, CAIP-97:
gp = [0.036420 0.248972 0.429217 0.248972 0.036420];
gd = [-0.108415 -0.280353 0 0.280353 0.108415];

dt = corrDn(corrDn(frame2-frame1, gp,'dont-compute'), gp','dont-compute');

bt = (frame1 + frame2)/2;  % average
dx = corrDn(corrDn(bt, gd,'dont-compute'), gp', 'dont-compute');
dy = corrDn(corrDn(bt, gp,'dont-compute'), gd','dont-compute');

clear bt

%  - -           -  -
% |a b|^{-1} =  |c -b| * 1/(ac-b^2)
% |b c|         |-b a|
%  - -           -  -

a = dx .^2;
b = dx .* dy;
c = dy .^ 2;
d = dt .* dx;
e = dt .* dy;

clear dx dy dt

% Blur (weighted sum of constraints):
if ischar(blur)
  if (strcmp(blur,'all'))
    a = mean2(a) + (sigp^2);
    b = mean2(b); 
    c = mean2(c) + (sigp^2);
    d = mean2(d); 
    e = mean2(e);
  end
else
  if (any(size(blur)==1))
    a = corrDn(corrDn(a, blur), blur') + (sigp^2);
    b = corrDn(corrDn(b, blur), blur');
    c = corrDn(corrDn(c, blur), blur') + (sigp^2);
    d = corrDn(corrDn(d, blur), blur');
    e = corrDn(corrDn(e, blur), blur');
  else    
    a = corrDn(a, blur) + (sigp^2);
    b = corrDn(b, blur);
    c = corrDn(c, blur) + (sigp^2);
    d = corrDn(d, blur);
    e = corrDn(e, blur);
  end
end

%  - -       -  -     - -
% |vx | = - |c -b| * | d | * 1/(ac-b^2)
% |vy |     |-b a|   | e |
%  - -       -  -     - -

denom = (a .* c) - (b .^ 2);

vx = ((b .* e) - (c .* d)) ./ denom;
vy = ((b .* d) - (a .* e)) ./ denom;

