% [newX, Mx, My] = adjustCorr2(X, Cx, Y, Cxy, MODE) 
%
% Linearly adjust variables in X to have correlation Cx, and cross-correlation Cxy.
% Rows of X, Y, and newX are samples of (random) row-vectors, such that:
%   1:  newX = X * Mx + Y * My
%   2:  newX' * newX = Cx
%   3:  newX' * Y = Cxy
%
% MODE is optional:
%   0 => choose randomly from the space of linear solutions
%   1 => simplest soln
%   2 => minimize angle change (DEFAULT) 
%   3 => Simple rotational (like adjustCorr1)
%   4 => SVD minimal vector change soln

% EPS, 11/25/97

function [newX, Mx, My] = adjustCorr2(X, Cx, Y, Cxy, mode)

if (exist('mode') ~= 1)
  mode = 2;
end

Bx = innerProd(X) / size(X,1);
Bxy = (X' * Y) / size(X,1);
By = innerProd(Y) / size(X,1);
iBy = inv(By);

Current = Bx - (Bxy * iBy * Bxy');
Desired = Cx - (Cxy * iBy * Cxy');

[E, D] = eig(Current);
D = diag(D);
if any(D < 0)
  fprintf(1,'Warning: negative eigenvalues:\n'); D'
end
[junk,Ind] = sort(D);
D = diag(sqrt(D(Ind(size(Ind,1):-1:1))));
E = E(:,Ind(size(Ind,1):-1:1));

[Eo,Do] = eig(Desired);
Do = diag(Do);
if any(Do < 0)
  fprintf(1,'Warning: negative eigenvalues:\n'); Do'
end
[junk,Ind] = sort(Do);
Do = diag(sqrt(Do(Ind(size(Ind,1):-1:1))));
Eo = Eo(:,Ind(size(Ind,1):-1:1));

if (mode == 0)
  Orth = orth(rand(size(D)));
elseif (mode == 1) % eye
  Orth = eye(size(D));
elseif (mode == 2) % simple
  A = [ eye(size(Cx)); -iBy*Bxy' ];
  Ao =  [ eye(size(Cx)); -iBy*Cxy' ];
  [U,S,V] = svd(E' * pinv(A) * Ao * Eo);
  Orth = U * V';
elseif (mode == 3)
  Orth = E' * Eo;
else     % SVD
  A = [ eye(size(Cx)); -iBy*Bxy' ];
  Ao =  [ eye(size(Cx)); -iBy*Cxy' ];
  [U,S,V] = svd(D * E' * pinv(A) * Ao * Eo * inv(Do));
  Orth = U * V';
end

Mx =  E * inv(D) * Orth * Do * Eo';
My =  iBy * (Cxy' - Bxy' * Mx);
newX = X * Mx + Y * My;
