%%%
%%% Q_SQUAREWAVE: MAKE A SQUARE-WAVE GRATING.
%%%
%%% 		Create a two-dimensional anti-aliased square-wave
%%% 		grating.  Control the size, frequency, phase,
%%% 		orientation, and duty-cycle.  The anti-aliasing is
%%% 		accomplished with a raised cosine transition function
%%% 		with a fixed transition width.
%%% 
%%% DATE: June 3, 1998
%%% BY:   Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_squarewave( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent, IM );
	Q_callbacks( parent );
	Q_squarewave( str2num(sprintf('%d',parent)), 0, 1 );
else 
	if( mode == 1 ) % PREVIEW
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		s	= min( 1, 128 / max( vals(1), vals(2) ) );
		squarewave= make_squarewave( s*vals(1), s*vals(2), vals(3), ...
					     vals(4), vals(5), vals(6) );
		Q_preview_im( squarewave, parent, 0, gray(256), [-1 1] );
	elseif( mode == 2 ) % OK
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		squarewave= make_squarewave( vals(1), vals(2), vals(3), ...
					     vals(4), vals(5), vals(6)  );
		Q_resetfig( parent, IM );
		Q_main( squarewave, gray(256), [-1 1], 1, 'gray','squarewave');
		return;
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Disc', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM )

	H 		= Q_get_handles( parent );
	[ydim,xdim] 	= size( IM.im );
	
	%%% SINUSOID XDIM/YDIM EDIT/LABEL 
	H.temp(1) = Q_input( parent, [0.72 0.37 0.1 0.06], xdim, 'edit' );
	H.temp(2) = Q_input( parent, [0.84 0.37 0.1 0.06], ydim, 'edit' );
	H.temp(3) = Q_input( parent, [0.5 0.37 0.2 0.06], 'X/Y size', 'text' );
	set( H.temp(3), 'FontSize', 12 );
	set( H.temp(3), 'HorizontalAlignment', 'right' );

	%%% SINUSOID FREQUENCY EDIT/LABEL
	H.temp(4) = Q_input( parent, [0.72 0.3 0.1 0.06], 1, 'edit' );
	H.temp(5) = Q_input( parent, [0.5 0.3 0.2 0.06], 'Frequency', 'text' );
	set( H.temp(5), 'FontSize', 12 );
	set( H.temp(5), 'HorizontalAlignment', 'right' );

	%%% SINUSOID FREQUENCY UNITS LABEL
	H.temp(6) = Q_input( parent, [0.84 0.3 0.1 0.06], '(cycles)', 'text' );
	set( H.temp(6), 'FontSize', 12 );
	set( H.temp(6), 'HorizontalAlignment', 'left' );

	%%% SINUSOID PHASE EDIT/LABEL
	H.temp(7) = Q_input( parent, [0.72 0.23 0.1 0.06], 0, 'edit' );
	H.temp(8) = Q_input( parent, [0.5 0.23 0.2 0.06], 'Phase', 'text' );
	set( H.temp(8), 'FontSize', 12 );
	set( H.temp(8), 'HorizontalAlignment', 'right' );

	%%% SINUSOID PHASE UNITS LABEL
	H.temp(9) = Q_input( parent, [0.84 0.23 0.1 0.06], '(rad)', 'text' );
	set( H.temp(9), 'FontSize', 12 );
	set( H.temp(9), 'HorizontalAlignment', 'left' );

	%%% SINUSOID ORIENTATION EDIT/LABEL
	H.temp(10) = Q_input( parent, [0.72 0.16 0.1 0.06], 0, 'edit' );
	H.temp(11) = Q_input( parent, [0.5 0.16 0.2 0.06],'Orientation','text');
	set( H.temp(11), 'FontSize', 12 );
	set( H.temp(11), 'HorizontalAlignment', 'right' );

	%%% SINUSOID ORIENTATION UNITS LABEL
	H.temp(12) = Q_input( parent, [0.84 0.16 0.1 0.06], '(deg)', 'text' );
	set( H.temp(12), 'FontSize', 12 );
	set( H.temp(12), 'HorizontalAlignment', 'left' );

	%%% SINUSOID DUTY CYCLE EDIT/LABEL
	H.temp(13) = Q_input( parent, [0.72 0.09 0.1 0.06], 0.5, 'edit' );
	H.temp(14) = Q_input( parent, [0.5 0.09 0.2 0.06],'Duty Cycle','text');
	set( H.temp(14), 'FontSize', 12 );
	set( H.temp(14), 'HorizontalAlignment', 'right' );

	%%% SINUSOID DUTY CYCLE UNITS LABEL
	H.temp(15) = Q_input( parent, [0.84 0.23 0.1 0.06], '(ratio)', 'text' );
	set( H.temp(15), 'FontSize', 12 );
	set( H.temp(15), 'HorizontalAlignment', 'left' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_squarewave(%d,0,1);', parent );
	cmd2 	= sprintf( 'Q_squarewave(%d,0,2);', parent );

	%%% SINUSOID XDIM/YDIM/FREQUENCY/PHASE/ORIENTATION/DUTY CYCLE
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd1 );
	set( H.temp(4), 'Callback', cmd1 );
	set( H.temp(7), 'Callback', cmd1 );
	set( H.temp(10), 'Callback', cmd1 );
	set( H.temp(13), 'Callback', cmd1 );
	set( H.ok, 'Callback', cmd2 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, IM )

	H 	= Q_get_handles( parent );
	[ydim2,xdim2] = size( IM.im ); % defaults

	xdim	= round( str2num( get( H.temp(1), 'String' ) ) );
	if( isempty(xdim) ) xdim = xdim2; end;
	xdim	= max( xdim, 8 );
	xdim	= min( xdim, 1024 );
	set( H.temp(1), 'String', xdim );

	ydim	= round( str2num( get( H.temp(2), 'String' ) ) );
	if( isempty(ydim) ) ydim = ydim2; end;
	ydim	= max( ydim, 8 );
	ydim	= min( ydim, 1024 );
	set( H.temp(2), 'String', ydim );

	freq	= str2num( get( H.temp(4), 'String' ) );
	if( isempty(freq) ) freq = 1; end;
	freq	= max( freq, 0 );
	set( H.temp(4), 'String', freq );

	phase	= str2num( get( H.temp(7), 'String' ) );
	if( isempty(phase) ) phase = 0; end;
	phase	= max( phase, 0 );
	phase	= min( phase, 2*pi );
	set( H.temp(7), 'String', phase );

	orient	= str2num( get( H.temp(10), 'String' ) );
	if( isempty(orient) ) orient = 0; end;
	orient	= max( orient, 0 );
	orient	= min( orient, 360 );
	set( H.temp(10), 'String', orient );

	duty	= str2num( get( H.temp(13), 'String' ) );
	if( isempty(duty) ) duty = 0.5; end;
	duty	= max( duty, 0.1 );
	duty	= min( duty, 0.9 );
	set( H.temp(13), 'String', duty );

	vals 	= [xdim ydim freq phase orient duty];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ square ] = make_squarewave( xdim, ydim, freq, phase, orient, duty )

	xdim		= round( xdim );
	ydim		= round( ydim );
	twidth		= 24;
        origin  	= [xdim/2 ydim/2];     
        [xramp,yramp] 	= meshgrid( 2*pi/xdim*([1:xdim]-origin(1)), ...
				   2*pi/ydim*([1:ydim]-origin(2)) );
	ramp		= cos(orient*pi/180)*xramp + sin(orient*pi/180)*yramp;
	square		= sin( freq*ramp + phase );
   
    	dim		= 255;
	d2		= ceil( dim/2 );
	X		= 2/dim * [-dim/2 : dim/2];
	midX		= floor( duty*(dim+1) );
	Y               = zeros( 1, dim+1 );
	Y(1:midX-twidth/2) = -1*ones(1,midX-twidth/2);
	Y(1+midX+twidth/2:dim+1) = 1*ones(1,1+dim-midX-twidth/2);
	Y(1+midX-twidth/2:midX+twidth/2) = sin( pi*[-0.5:1/(twidth-1):0.5] );
	square		= reshape( interp1(X,Y,square(:),'*linear'),ydim,xdim );
