%%%
%%% Q_SINUSOID: MAKE A SINUSOIDAL GRATING.
%%%
%%%		Create a two-dimensional sinusoidal grating.  Control
%%%		the size, frequency, phase, and orientation.
%%%
%%% DATE: 	June 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_sinusoid( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent, IM );
	Q_callbacks( parent );
	Q_sinusoid( str2num(sprintf('%d',parent)), 0, 1 );
else 
	if( mode == 1 ) % PREVIEW
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		s	= min( 1, 128 / max( vals(1), vals(2) ) );
		sinusoid= make_sinusoid( s*vals(1), s*vals(2), vals(3), ...
					 vals(4), vals(5) );
		Q_preview_im( sinusoid, parent, 0, gray(256), [-1 1] );
	elseif( mode == 2 ) % OK
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		sinusoid= make_sinusoid( vals(1), vals(2), vals(3), ...
					 vals(4), vals(5)  );
		Q_resetfig( parent, IM );
		Q_main( sinusoid, gray(256), [-1 1], 1, 'gray', 'sinusoid' );
		return;
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Disc', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM )

	H 		= Q_get_handles( parent );
	[ydim,xdim] 	= size( IM.im );
	
	%%% XDIM/YDIM EDIT/LABEL 
	H.temp(1) = Q_input( parent, [0.72 0.37 0.1 0.06], xdim, 'edit' );
	H.temp(2) = Q_input( parent, [0.84 0.37 0.1 0.06], ydim, 'edit' );
	H.temp(3) = Q_input( parent, [0.5 0.37 0.2 0.06], 'X/Y size', 'text' );
	set( H.temp(3), 'FontSize', 12 );
	set( H.temp(3), 'HorizontalAlignment', 'right' );

	%%% FREQUENCY EDIT/LABEL
	H.temp(4) = Q_input( parent, [0.72 0.3 0.1 0.06], 1, 'edit' );
	H.temp(5) = Q_input( parent, [0.5 0.3 0.2 0.06], 'Frequency', 'text' );
	set( H.temp(5), 'FontSize', 12 );
	set( H.temp(5), 'HorizontalAlignment', 'right' );

	%%% FREQUENCY UNITS LABEL
	H.temp(6) = Q_input( parent, [0.84 0.3 0.1 0.06], '(cycles)', 'text' );
	set( H.temp(6), 'FontSize', 12 );
	set( H.temp(6), 'HorizontalAlignment', 'left' );

	%%% PHASE EDIT/LABEL
	H.temp(7) = Q_input( parent, [0.72 0.23 0.1 0.06], 0, 'edit' );
	H.temp(8) = Q_input( parent, [0.5 0.23 0.2 0.06], 'Phase', 'text' );
	set( H.temp(8), 'FontSize', 12 );
	set( H.temp(8), 'HorizontalAlignment', 'right' );

	%%% PHASE UNITS LABEL
	H.temp(9) = Q_input( parent, [0.84 0.23 0.1 0.06], '(rad)', 'text' );
	set( H.temp(9), 'FontSize', 12 );
	set( H.temp(9), 'HorizontalAlignment', 'left' );

	%%% ORIENTATION EDIT/LABEL
	H.temp(10) = Q_input( parent, [0.72 0.16 0.1 0.06], 0, 'edit' );
	H.temp(11) = Q_input( parent,[0.5 0.16 0.2 0.06],'Orientation','text');
	set( H.temp(11), 'FontSize', 12 );
	set( H.temp(11), 'HorizontalAlignment', 'right' );

	%%% ORIENTATION UNITS LABEL
	H.temp(12) = Q_input( parent, [0.84 0.16 0.1 0.06], '(deg)', 'text' );
	set( H.temp(12), 'FontSize', 12 );
	set( H.temp(12), 'HorizontalAlignment', 'left' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_sinusoid(%d,0,1);', parent );
	cmd2 	= sprintf( 'Q_sinusoid(%d,0,2);', parent );

	%%% XDIM/YDIM/FREQUENCY/PHASE/ORIENTATION
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd1 );
	set( H.temp(4), 'Callback', cmd1 );
	set( H.temp(7), 'Callback', cmd1 );
	set( H.temp(10), 'Callback', cmd1 );
	set( H.ok, 'Callback', cmd2 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, IM )

	H 	= Q_get_handles( parent );
	[ydim2,xdim2] = size( IM.im ); % defaults

	xdim	= round( str2num( get( H.temp(1), 'String' ) ) );
	if( isempty(xdim) ) xdim = xdim2; end;
	xdim	= max( xdim, 8 );
	xdim	= min( xdim, 1024 );
	set( H.temp(1), 'String', xdim );

	ydim	= round( str2num( get( H.temp(2), 'String' ) ) );
	if( isempty(ydim) ) ydim = ydim2; end;
	ydim	= max( ydim, 8 );
	ydim	= min( ydim, 1024 );
	set( H.temp(2), 'String', ydim );

	freq	= str2num( get( H.temp(4), 'String' ) );
	if( isempty(freq) ) freq = 1; end;
	freq	= max( freq, 0 );
	set( H.temp(4), 'String', freq );

	phase	= str2num( get( H.temp(7), 'String' ) );
	if( isempty(phase) ) phase = 0; end;
	phase	= max( phase, 0 );
	phase	= min( phase, 2*pi );
	set( H.temp(7), 'String', phase );

	orient	= str2num( get( H.temp(10), 'String' ) );
	if( isempty(orient) ) orient = 0; end;
	orient	= max( orient, 0 );
	orient	= min( orient, 360 );
	set( H.temp(10), 'String', orient );

	vals 	= [xdim ydim freq phase orient];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ sine ] = make_sinusoid( xdim, ydim, freq, phase, orient )

        origin  	= [xdim/2 ydim/2];     
        [xramp,yramp] 	= meshgrid( 2*pi/xdim*([1:xdim]-origin(1)), ...
				   2*pi/ydim*([1:ydim]-origin(2)) );
	orient		= orient * pi/180;
	ramp		= cos(orient)*xramp + sin(orient)*yramp;
	sine		= sin( freq*ramp + phase );