%%%
%%% Q_ROTATEFREE: ROTATE IMAGE.
%%%
%%%		Rotate image by an aribtrary amount.  Previewing is
%%%		done with fast linear interpolation, and final
%%%		rotation is done with slower bicubic interpolation.
%%%		While previewing, rotations are always done with
%%%		respect to initial image. 
%%%
%%% DATE: 	July 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_rotatefree( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent );
	Q_callbacks( parent );
	Q_rotatefree( str2num(sprintf('%d',parent)), 0, 1 );
else 
	if( mode == 1 | mode == 2 ) % PREVIEW
		vals	= Q_getvals( parent, mode );	
		IM	= Q_get_iminfo( parent );
		im2	= rotate( Q_make_preview_im(IM.im), vals, '*linear' );
		Q_preview_im( im2, parent, 0 );
	elseif( mode == 3 ) % OK
		vals	= Q_getvals( parent, mode );	
		IM	= Q_get_iminfo( parent );
		IM.im	= rotate( IM.im, vals, '*cubic' );
		Q_resetfig( parent, IM );
		return;
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Rotate Free', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent )

	H 	= Q_get_handles( parent );

	%%% SLIDER/EDIT/LABEL
	H.temp(1) = Q_input( parent, [0.6 0.37 0.25 0.06], ...
				[-360 360 0 0.0028 0.014], 'slider' );
	H.temp(2) = Q_input( parent, [0.86 0.37 0.12 0.06], 0, 'edit' );
	H.temp(3) = Q_input( parent, [0.86 0.30 0.12 0.06], ...
				'Angle (deg)', 'text' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_rotatefree(%d,0,1);', parent );
	cmd2 	= sprintf( 'Q_rotatefree(%d,0,2);', parent );
	cmd3 	= sprintf( 'Q_rotatefree(%d,0,3);', parent );

	%%% SLIDER/EDIT
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd2 );
	set( H.ok, 'Callback', cmd3 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, mode )

	H	= Q_get_handles( parent );

	if( mode == 1 | mode == 3 | mode == 4 ) % SLIDER INPUT
		r	= round( get( H.temp(1), 'Value' ) );
		set(  H.temp(2), 'String', r );
	elseif( mode == 2 ) % EDIT INPUT
		r	= str2num( get( H.temp(2), 'String' ) );
		if( isempty(r) ) r = 0; end;
		r	= max( r, get(  H.temp(1), 'Min' ) );
		r	= min( r, get(  H.temp(1), 'Max' ) );
		set( H.temp(1), 'Value', r );
		set( H.temp(2), 'String', r );
	end

	vals 	= r;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ im2 ] = rotate( im, th, method )
	
	[ydim,xdim] = size( im );
	midx	= round( xdim/2 );
	midy	= round( ydim/2 );
	th	= th * pi/180;
	
	if( th == 0 )
		im2 = im;
		return;
	end

	%%% ROTATE
	xramp	 = ones( ydim, 1 ) * [1 : xdim];
	yramp	 = [1 : ydim]' * ones( 1, xdim );
	warpx	 = midx + (cos(th) * (xramp-midx) - sin(th)*(yramp-midy));
	warpy	 = midy + (sin(th) * (xramp-midx) + cos(th)*(yramp-midy));
	im2	 = interp2( xramp, yramp, im, warpx, warpy, method );

	%%% REMOVE NANS DUE TO INTERP2
	im2	 = im2(:);
	index	 = find( isnan( im2 ));
	im2(index) = zeros( 1, size(index,1) );
	im2	 = reshape( im2, ydim, xdim );

