%%%
%%% Q_RECTPOLAR: RECTANGULAR <-> POLAR CONVERSION
%%%
%%%		Convert from rectangular to polar coordinates and
%%%		back.  Both conversions appear automatically, select
%%%		check box of desired conversion(s) and choose OK.  New
%%%		Q figure is created for converted image(s).  Out of
%%%		range values are replaced with 0. 
%%%
%%% DATE: 	July 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_rectpolar( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent );
	Q_callbacks( parent );
else 
	if( mode == 1 ) % OK
		IM	= Q_get_iminfo( parent );
		H	= Q_get_handles( parent );
		vals	= Q_getvals( parent );
		if( any(vals) )
			[r2p,p2r] = rectpolar( IM.im, '*cubic' );
		end
		Q_resetfig( parent, IM );
		if( vals(2) )	% POLAR TO RECT
			Q_main( p2r, IM.cmap, IM.clim, IM.magnify, IM.mode', ...
				sprintf( '%s_rect', IM.filename ) );
		end
		if( vals(1) )	% RECT TO POLAR
			Q_main( r2p, IM.cmap, IM.clim, IM.magnify, IM.mode', ...
				sprintf( '%s_polar', IM.filename ) );
		end
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Rect/Polar', 1, 1 );

	%%% COMPUTE BOTH CONVERSIONS AND DISPLAY
	IM		= Q_get_iminfo( parent );
	[r2p,p2r]	= rectpolar( Q_make_preview_im(IM.im), '*linear' );

	Q_preview_im( r2p, parent, 0, -1, -1, [0.53 0.05 0.35 0.35] );
	Q_preview_im( p2r, parent, 0 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent )

	H 	= Q_get_handles( parent );
	
	%%% CONVERSION CHECKBOXES
	H.temp(1) = Q_input( parent, [0.82 0.05 0.16 0.06], ...
				'Rect->Polar', 'checkbox' );
	H.temp(2) = Q_input( parent, [0.82 0.55 0.16 0.06], ...
				'Polar->Rect', 'checkbox' );

	set( H.temp(1), 'Value', 0 );
	set( H.temp(2), 'Value', 0 );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_rectpolar(%d,0,1);', parent );
	set( H.ok, 'Callback', cmd1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent )

	H 	= Q_get_handles( parent );
	vals	= [ get(H.temp(1),'Value') get(H.temp(2),'Value') ]; 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ r2p, p2r ] = rectpolar( im, method )
	
	[ydim,xdim] = size( im );
	midx	= round( xdim/2 );
	midy	= round( ydim/2 );
	
	%%% RECT TO POLAR
        x1	= ones( ydim,1) * [1:xdim] - midx;
        y1	= [1:ydim]' * ones(1,xdim) - midy;

        r	= sqrt( x1.^2 + y1.^2 );
        r	= r - min(r(:));
        r	= r * sqrt(2) * xdim/max(r(:));
        th	= atan2( x1, y1 );
        th	= th - min(th(:));
        th	= th * ydim/max(th(:));
        r2p	= interp2( y1+midy, x1+midx, im, th, r, method );
        
	%%% POLAR TO RECT
	r	= ones( ydim,1) * [1:xdim];
        th	= [1:ydim]' * ones(1,xdim);

        x	= r .* cos(pi - th * 2*pi/xdim);
        x	= x - min(x(:));
        x	= x * xdim/max(x(:));
        y	= r .* sin(pi - th * 2*pi/ydim);
        y	= y - min(y(:));
        y	= y * ydim/max(y(:));
        p2r	= interp2( th, r, im, y, x, method );
        p2r	= rot90( p2r, -1 ); % hack, fix this.

	%%% REMOVE NANS DUE TO INTERP2
	r2p	= r2p(:);
	index	= find( isnan( r2p ));
	r2p(index)= zeros( 1, size(index,1) );
	r2p	= reshape( r2p, ydim, xdim );
