%%%
%%% Q_MATLABCMD: EXECUTE A MATLAB COMMAND.
%%%
%%%		Execute an arbitrary matlab command and place results
%%%		in a new Q figure.  Command can include variables from
%%%		the global matlab environment, and images within a Q
%%%		figure.  For example, to add the matlab variable X and
%%%		the image in Q figure 1: X + {1}.  Combine two Q
%%%		images by: 0.5*{1} + 0.5*{2}.  Pass a Q image to an
%%%		arbitrary matlab function: abs( fft2({1}) ). 
%%%		
%%%		Results are previewed in a small window, click on the
%%%		Evaluate button to create a new Q figure while leaving
%%%		menu option open.  The default colormap/clim defined
%%%		in Q_defaults.m are used for displaying results. 
%%%
%%% DATE: 	June 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_matlabcmd( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	Q_window( parent );
	Q_sliders( parent );
	Q_callbacks( parent );
else
	if( mode == 1 ) % PREVIEW 
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, mode );
		im 	= parse_eval( parent, vals.name );
		im 	= Q_make_preview_im( im );
		cmap = Q_defaults( 'cmap' );
		clim = Q_defaults( 'clim' );
		Q_preview_im( im, parent, vals.auto, cmap, clim );
	elseif( mode == 2 ) % EVALUATE BUT DON'T CLOSE 
		vals	= Q_getvals( parent, mode );
		im 	= parse_eval( parent, vals.name );
		IM	= Q_get_iminfo( parent );
		D 	= Q_defaults( 'all' );
		if( vals.auto )
			D.clim = [min(im(:)) max(im(:))];
		end
		Q_main( im, D.cmap, D.clim, D.magnify, D.mode, vals.name );
	elseif( mode == 3 ) % OK 
		vals	= Q_getvals( parent, mode );
		im 	= parse_eval( parent, vals.name );
		IM	= Q_get_iminfo( parent );
		D 	= Q_defaults( 'all' );
		Q_resetfig( parent, IM );
		if( vals.auto )
			D.clim = [min(im(:)) max(im(:))];
		end
		Q_main( im, D.cmap, D.clim, D.magnify, D.mode, vals.name );
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Save', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent )

	H		= Q_get_handles( parent );

	%%% FILENAME ENTRY
	H.temp(1)	= Q_input( parent, [0.6 0.38 0.35 0.1], ...
				'', 'edit' );
	set( H.temp(1), 'FontSize', 12 );
	set( H.temp(1), 'HorizontalAlignment', 'left' );

	%%% FILENAME LABEL
	H.temp(2) 	= Q_input( parent, [0.6 0.25 0.35 0.1], ...
				'Matlab Command', 'text' );
	set( H.temp(2), 'FontSize', 12 );
	set( H.temp(2), 'HorizontalAlignment', 'left' );

	%%% AUTOSCALE OPTION
	H.temp(3) 	= Q_input( parent, [0.82 0.55 0.14 0.06], ...
				'Autoscale', 'checkbox' );

	%%% EVALUATE OPTION
	H.temp(4) 	= Q_input( parent, [0.82 0.65 0.14 0.06], ...
				'Evaluate', 'pushbutton' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H	= Q_get_handles( parent );
	cmd1	= sprintf( 'Q_matlabcmd( %d,0,1 );', parent );
	cmd2	= sprintf( 'Q_matlabcmd( %d,0,2 );', parent );
	cmd3	= sprintf( 'Q_matlabcmd( %d,0,3 );', parent );

	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(3), 'Callback', cmd1 );
	set( H.temp(4), 'Callback', cmd2 );
	set( H.ok, 'Callback', cmd3 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, mode )

	H		= Q_get_handles( parent );	
	vals.name	= get( H.temp(1), 'String' ); % filename
	vals.auto	= get( H.temp(3), 'Value' ); % autoscale option

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ im ] = parse_eval( parent, line )

	%%% PARSE
	vars	= ['Q_A' 'Q_B' 'Q_C' 'Q_D' 'Q_E' 'Q_F' 'Q_G' 'Q_H' 'Q_I' ...
		   'Q_J' 'Q_K' 'Q_L' 'Q_M' 'Q_N' 'Q_O' 'Q_P' 'Q_Q'];
	index1	= findstr( line, '{' );
	index2	= findstr( line, '}' );

	if( isempty(index1) & isempty(index2) )
		cmd = line;
	elseif( size(index1,2) ~= size(index2,2) )
		cmd = [];
	else
		cmd = line;
		c = 1;
		for i = 1 : size(index1,2)
			num	= str2num( line(index1(i)+1:index2(i)-1) );
			IM	= Q_get_iminfo( num );
			assignin( 'base', vars(c:c+2), IM.im );
			if( index2(i)-index1(i) == 2 )
				cmd( index1(i):index1(i)+2 ) = vars(c:c+2);
			else 
				cmd( index1(i):index1(i)+2 ) = vars(c:c+2);
				cmd( index2(i) )   = ' ';
			end
			c = c + 3;
		end
	end

	%%% EXECUTE
	if( isempty( line ) )
		IM = Q_get_iminfo( parent );
		im = IM.im;
	elseif( isempty( cmd ) )
		fprintf( 'Q_matlabcmd(): error parsing command\n' );
		IM = Q_get_iminfo( parent );
		im = IM.im;
	else
		im = evalin( 'base', cmd );
	end
	
	%%% CLEAR TEMPORARY GLOBALS
	c = 1;
	for i = 1 : size( index1,2 )
		evalin( 'base', sprintf( 'clear %s', vars(c:c+2) ) );
		c = c + 3;
	end
	return;



