%%%
%%% Q_LINE_PROFILE: PLOT IMAGE LINE PROFILE
%%%
%%%		Plot a one-dimensional image profile.  Click and drag
%%%		to control the line through the image.  Plots display
%%%		pixel values.  Linear interpolation is used on
%%%		non-horizontal or vertical lines.
%%%
%%% DATE: 	July 13, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function[] = Q_line_profile( parent, first, mode )

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent );
else 
	if( mode == 1 ) 	% BEGIN DRAWING LINE
		H = Q_get_handles( parent );
		[x,y] = get_xy( parent );
		set( H.temp(1), 'String', x );
		set( H.temp(2), 'String', y );
		set( H.temp(3), 'String', x );
		set( H.temp(4), 'String', y );
		cmd2 = sprintf( 'Q_line_profile( %d,0,2 );', parent );
		set( parent, 'WindowButtonMotionFcn', cmd2 );
	elseif( mode == 2 ) 	% DRAW LINE IN REAL-TIME
		draw_line( parent )
	elseif( mode == 3 ) 	% DONE DRAWING LINE - SHOW PROFILE
		set( parent, 'WindowButtonMotionFcn', '' );
		draw_profile( parent );
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Line Profile', 1, 0 );
	cmd1	= sprintf( 'Q_line_profile( %d,0,1 );', parent );
	cmd3	= sprintf( 'Q_line_profile( %d,0,3 );', parent );

        set( parent, 'WindowButtonDownFcn', cmd1 );
        set( parent, 'WindowButtonUpFcn', cmd3 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM )

	H 	= Q_get_handles( parent );

	%%% X1,Y1 / X2,Y2 EDIT
	H.temp(1) = Q_input( parent, [0.6 0.37 0.08 0.06], 0, 'edit' );
	H.temp(2) = Q_input( parent, [0.69 0.37 0.08 0.06], 0, 'edit' );
	H.temp(3) = Q_input( parent, [0.78 0.37 0.08 0.06], 0, 'edit' );
	H.temp(4) = Q_input( parent, [0.87 0.37 0.08 0.06], 0, 'edit' );

	%%% X1,Y1 / X2,Y2 LABEL
	H.temp(5) = Q_input( parent, [0.6 0.3 0.08 0.06], 'X1', 'text' );
	H.temp(6) = Q_input( parent, [0.69 0.3 0.08 0.06], 'Y1', 'text' );
	H.temp(7) = Q_input( parent, [0.78 0.3 0.08 0.06], 'X2', 'text' );
	H.temp(8) = Q_input( parent, [0.87 0.3 0.08 0.06], 'Y2', 'text' );

	%%% LINE HANDLE
	H.temp(9) = line( [0 0], [0 0] );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent )

	H	= Q_get_handles( parent );
	x0 	= str2num( get( H.temp(1), 'String' ) );
	y0 	= str2num( get( H.temp(2), 'String' ) );
	x1 	= str2num( get( H.temp(3), 'String' ) );
	y1 	= str2num( get( H.temp(4), 'String' ) );
	vals	= [x0 y0 x1 y1];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = draw_line( parent )

	H 	= Q_get_handles( parent );
	[x,y] 	= get_xy( parent );
	set( H.temp(3), 'String', x );
	set( H.temp(4), 'String', y );

	vals 	= Q_getvals( parent );
	set( H.temp(9), 'Xdata', [vals(1) vals(3)] );
	set( H.temp(9), 'Ydata', [vals(2) vals(4)] );
	set( H.temp(9), 'Color', [1 0 0] );
	set( H.temp(9), 'Visible', 'on' );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = draw_profile( parent )

	IM	= Q_get_iminfo( parent );
	H 	= Q_get_handles( parent );
	vals 	= Q_getvals( parent );
	x0 	= vals(1);
	y0 	= vals(2);
	x1 	= vals(3);
	y1 	= vals(4);
	X	= sort( [ x0 x1 ] );
	Y	= sort( [ y0 y1 ] );

	if( y1 == y0 & x1 == x0)
		figure( parent );
		subplot( 'position',  [0.6 0.55 0.37 0.35] );
		cla;
		set( H.temp(9), 'Visible', 'off' );
		return;
	elseif( y1 == y0 )
		slice = IM.im( Y(1), X(1):X(2) );
	elseif( x1 == x0 )
		slice = IM.im( Y(1):Y(2), X(1) )';
	else
		[ydim,xdim] = size( IM.im );
		xramp	= ones( ydim, 1 ) * [1 : xdim];
		yramp	= [1 : ydim]' * ones( 1, xdim );
		m	= (y1 - y0) / (x1 - x0); % equation of line
		b	= y1 - m*x1;		 % equation of line
		c 	= 1;
		for x = X(1) : (X(2)-X(1))/100 : X(2)
			y = m*x + b;
			slice(c) = interp2( xramp, yramp, IM.im, x, y, ...
					'*linear' );
			c = c + 1;
		end
	end

	figure( parent );
	subplot( 'position',  [0.6 0.55 0.37 0.35] );
	slice = 255 * Q_cmaplookup( IM.cmap, IM.clim, slice );
	plot( slice );
	axis( [1 size(slice,2) -5 260] );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Use instead of Q_get_xy because of expanded figure
function [x, y] = get_xy( parent ) 
	
	set( parent, 'Units', 'pixels' );
	H		= Q_get_handles( parent );
	IM		= Q_get_iminfo( parent );
	[ydim,xdim] 	= size(IM.im);
	fig_pos		= get( parent, 'Position' );
	im_dims		= [2*xdim ydim]; % flip x/y
	fig_dims	= [fig_pos(3) fig_pos(4)];
	scale		= im_dims ./ fig_dims;
	offset		= fig_dims - im_dims./max(scale);
	scale		= im_dims ./ (fig_dims-offset);
	mouse_pos	= get( parent, 'CurrentPoint' );
	x		= mouse_pos(1) + 1;
	x		= scale(1) * (x - offset(1)/4);
	y		= fig_dims(2) - mouse_pos(2) + 1;
	y		= scale(2) * (y - offset(2)/2);
	x		= round( max( min(x,xdim), 1 ) );
	y		= round( max( min(y,ydim), 1 ) );
	set( parent, 'Units', 'normalized' );
