%%%
%%% Q_GAUSSIAN: RENDER A GAUSSIAN IMAGE.
%%%
%%%		Make a two-dimensional Gaussian image.  Control the
%%%		horizontal and vertical dimensions and variances.
%%%		Final image is normalized in the range [0,1]. 
%%%
%%% DATE: 	June 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_gaussian( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent, IM );
	Q_callbacks( parent );
	Q_gaussian( str2num(sprintf('%d',parent)), 0, 1 );
else 
	if( mode == 1 ) % PREVIEW
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		s	= min( 1, 128 / max( vals(1), vals(2) ) );
		gaussian= make_gaussian( s*vals(1), s*vals(2), ...
					 s*vals(3), s*vals(4) );
		Q_preview_im( gaussian, parent, 0, gray(256), [0 1] );
	elseif( mode == 2 ) % OK
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		gaussian= make_gaussian( vals(1), vals(2), vals(3), vals(4) );
		Q_resetfig( parent, IM );
		Q_main( gaussian, gray(256), [0 1], 1, 'gray', 'Gaussian' );
		return;
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Gaussian', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM )

	H 		= Q_get_handles( parent );
	[ydim,xdim] 	= size( IM.im );
	xcov		= min( xdim, ydim ) / 4;
	ycov 		= xcov;
	
	%%% XDIM/YDIM EDIT/LABEL 
	H.temp(1) = Q_input( parent, [0.72 0.37 0.1 0.06], xdim, 'edit' );
	H.temp(2) = Q_input( parent, [0.84 0.37 0.1 0.06], ydim, 'edit' );
	H.temp(3) = Q_input( parent, [0.5 0.37 0.2 0.06], 'X/Y size', 'text' );
	set( H.temp(3), 'FontSize', 12 );
	set( H.temp(3), 'HorizontalAlignment', 'right' );

	%%% XCOV/YCOV EDIT/LABEL 
	H.temp(4) = Q_input( parent, [0.72 0.3 0.1 0.06], xcov, 'edit' );
	H.temp(5) = Q_input( parent, [0.84 0.3 0.1 0.06], ycov, 'edit' );
	H.temp(6) = Q_input( parent, [0.5 0.3 0.2 0.06], 'X/Y cov', 'text' );
	set( H.temp(6), 'FontSize', 12 );
	set( H.temp(6), 'HorizontalAlignment', 'right' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_gaussian(%d,0,1);', parent );
	cmd2 	= sprintf( 'Q_gaussian(%d,0,2);', parent );

	%%% XDIM/YDIM/XCOV/YCOV
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd1 );
	set( H.temp(4), 'Callback', cmd1 );
	set( H.temp(5), 'Callback', cmd1 );
	set( H.ok, 'Callback', cmd2 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, IM )

	H 	= Q_get_handles( parent );
	[ydim2,xdim2] = size( IM.im ); % defaults
	xcov2	= min( xdim2, ydim2 ) / 4; % defaults
	ycov2 	= xcov2; % defaults

	xdim	= round( str2num( get( H.temp(1), 'String' ) ) );
	if( isempty(xdim) ) xdim = xdim2; end;
	xdim	= max( xdim, 8 );
	xdim	= min( xdim, 1024 );
	set( H.temp(1), 'String', xdim );

	ydim	= round( str2num( get( H.temp(2), 'String' ) ) );
	if( isempty(ydim) ) ydim = ydim2; end;
	ydim	= max( ydim, 8 );
	ydim	= min( ydim, 1024 );
	set( H.temp(2), 'String', ydim );

	xcov	= str2num( get( H.temp(4), 'String' ) );
	if( isempty(xcov) ) xcov = xcov2; end;
	xcov	= max( xcov, 1 );
	set( H.temp(4), 'String', xcov );

	ycov	= str2num( get( H.temp(5), 'String' ) );
	if( isempty(ycov) ) ycov = ycov2; end;
	ycov	= max( ycov, 1 );
	set( H.temp(5), 'String', ycov );

	vals 	= [xdim ydim xcov ycov];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ gauss ] = make_gaussian( xdim, ydim, xcov, ycov )
	
        origin  	= [xdim/2 ydim/2];     
        [xramp,yramp] 	= meshgrid( [1:xdim]-origin(1), [1:ydim]-origin(2) );
	gauss_norm	= 1/(2*pi*sqrt(xcov*ycov));
	gauss		= gauss_norm * ...
			   exp( -xramp.^2/(2*xcov^2 ) - yramp.^2/(2*ycov^2 ) );
	gauss		= gauss - min(gauss(:));
	gauss		= gauss / max(gauss(:));