%%%
%%% Q_FRACTAL:	RENDER A FRACTAL IMAGE.
%%%
%%%		Make a fractal image with a power spectral response of
%%%		the form: 
%%%			F(w) = 1/w^(5-2*p)
%%%
%%%		Control the spectral spread with the parameter p in [0,4].
%%%		Final image is normalized to have unit variance in the
%%%		range [0,1]. 
%%%
%%% DATE: 	June 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_fractal( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent, IM );
	Q_callbacks( parent );
	Q_fractal( str2num(sprintf('%d',parent)), 0, 1 );
else 
	if( mode == 1 ) % PREVIEW
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		s	= min( 1, 128 / max( vals(1), vals(2) ) );
		fractal	= make_fractal( s*vals(1), s*vals(2), vals(3) );
		Q_preview_im( fractal, parent, 0, gray(256), [0 1] );
	elseif( mode == 2 ) % OK
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		fractal	= make_fractal( vals(1), vals(2), vals(3) );
		Q_resetfig( parent, IM );
		Q_main( fractal, gray(256), [0 1], 1, 'gray', 'fractal' );
		return;
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Fractal', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM )

	H 		= Q_get_handles( parent );
	[ydim,xdim] 	= size( IM.im );
	
	%%% XDIM/YDIM EDIT/LABEL 
	H.temp(1) = Q_input( parent, [0.72 0.37 0.1 0.06], xdim, 'edit' );
	H.temp(2) = Q_input( parent, [0.84 0.37 0.1 0.06], ydim, 'edit' );
	H.temp(3) = Q_input( parent, [0.5 0.37 0.2 0.06], 'X/Y size', 'text' );
	set( H.temp(3), 'FontSize', 12 );
	set( H.temp(3), 'HorizontalAlignment', 'right' );

	%%% POWER EDIT/LABEL
	H.temp(4) = Q_input( parent, [0.72 0.3 0.1 0.06], 0.5, 'edit' );
	H.temp(5) = Q_input( parent, [0.5 0.3 0.2 0.06], 'Power', 'text' );
	set( H.temp(5), 'FontSize', 12 );
	set( H.temp(5), 'HorizontalAlignment', 'right' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_fractal(%d,0,1);', parent );
	cmd2 	= sprintf( 'Q_fractal(%d,0,2);', parent );

	%%% XDIM/YDIM/POWER
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd1 );
	set( H.temp(4), 'Callback', cmd1 );
	set( H.ok, 'Callback', cmd2 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, IM )

	H 	= Q_get_handles( parent );
	[ydim2,xdim2] = size( IM.im ); % defaults

	xdim	= round( str2num( get( H.temp(1), 'String' ) ) );
	if( isempty(xdim) ) xdim = xdim2; end;
	xdim	= max( xdim, 8 );
	xdim	= min( xdim, 1024 );
	set( H.temp(1), 'String', xdim );

	ydim	= round( str2num( get( H.temp(2), 'String' ) ) );
	if( isempty(ydim) ) ydim = ydim2; end;
	ydim	= max( ydim, 8 );
	ydim	= min( ydim, 1024 );
	set( H.temp(2), 'String', ydim );

	power	= str2num( get( H.temp(4), 'String' ) );
	if( isempty(power) ) power = 0.5; end;
	power	= max( power, 0 );
	power	= min( power, 4 );
	set( H.temp(4), 'String', power );

	vals 	= [xdim ydim power];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ res ] = make_fractal( xdim, ydim, pow )

	xdim	= round( xdim );
	ydim	= round( ydim );
	res 	= randn( ydim, xdim );
	fres 	= fft2( res );
	sz 	= size( res );
	ctr 	= ceil( (sz+1)./2 );
	[xramp,yramp] = meshgrid( [1:sz(2)]-ctr(2), [1:sz(1)]-ctr(1) );
	expt	= -( 2.5-pow );
	rad 	= ( xramp.^2 + yramp.^2 ) .^ ( expt/2 );
	shape 	= fftshift( rad );
	shape(1,1) = 1;
	fres 	= shape .* fres;
	fres 	= ifft2( fres );
	if( max(abs(imag(fres(:)))) > 1e-10 )
  		fprintf( 'Q_fractal(): Symmetry error in creating fractal\n' );
		res = zeros( ydim, xdim );
		return;
	else
  		res = real(fres);
		mn  = mean( res(:) );
  		res = res/sqrt( sum(abs(res(:)-mn).^2)/(prod(size(res)) - 1));
	end
	res	= res - min(res(:));
	res	= res / max(res(:));

