%%%
%%% Q_DISC: 	RENDER A DISC.
%%%
%%%		Make an anti-aliased disc with value 1 in the center
%%%		and value 0 on the outside.  Control the size of the
%%%		image and the radius of the disc.  The radius can't be
%%%		larger than the smallest image dimension. The
%%%		transition function is a raised cosine with a fixed
%%%		transition width. 
%%%
%%% DATE: 	June 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_disc( parent, first, mode );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	IM	= Q_get_iminfo( parent );
	Q_window( parent );
	Q_sliders( parent, IM );
	Q_callbacks( parent );
	Q_disc( str2num(sprintf('%d',parent)), 0, 1 );
else 
	if( mode == 1 ) % PREVIEW
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		s	= min( 1, 128 / max( vals(1), vals(2) ) );
		disc	= make_disc( s*vals(1), s*vals(2), s*vals(3) );
		Q_preview_im( disc, parent, 0, gray(256), [0 1] );
	elseif( mode == 2 ) % OK
		IM	= Q_get_iminfo( parent );
		vals	= Q_getvals( parent, IM );	
		disc	= make_disc( vals(1), vals(2), vals(3) );
		Q_resetfig( parent, IM );
		Q_main( disc, gray(256), [0 1], 1, 'gray', 'disc' );
		return;
	end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent )

	Q_expandfig( parent, 'Disc', 1, 1 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM )

	H 		= Q_get_handles( parent );
	[ydim,xdim] 	= size( IM.im );
	radius		= min( xdim, ydim ) / 3.2;
	
	%%% XDIM/YDIM EDIT/LABEL 
	H.temp(1) = Q_input( parent, [0.72 0.37 0.1 0.06], xdim, 'edit' );
	H.temp(2) = Q_input( parent, [0.84 0.37 0.1 0.06], ydim, 'edit' );
	H.temp(3) = Q_input( parent, [0.5 0.37 0.2 0.06], 'X/Y size', 'text' );
	set( H.temp(3), 'FontSize', 12 );
	set( H.temp(3), 'HorizontalAlignment', 'right' );

	%%% RADIUS EDIT/LABEL
	H.temp(4) = Q_input( parent, [0.72 0.3 0.1 0.06], radius, 'edit' );
	H.temp(5) = Q_input( parent, [0.5 0.3 0.2 0.06], 'Radius', 'text' );
	set( H.temp(5), 'FontSize', 12 );
	set( H.temp(5), 'HorizontalAlignment', 'right' );

	%%% RADIUS UNITS LABEL
	H.temp(6) = Q_input( parent, [0.84 0.3 0.1 0.06], '(pixels)', 'text' );
	set( H.temp(6), 'FontSize', 12 );
	set( H.temp(6), 'HorizontalAlignment', 'left' );

	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_disc(%d,0,1);', parent );
	cmd2 	= sprintf( 'Q_disc(%d,0,2);', parent );

	%%% XDIM/YDIM/RADIUS
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd1 );
	set( H.temp(4), 'Callback', cmd1 );
	set( H.ok, 'Callback', cmd2 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, IM )

	H 	= Q_get_handles( parent );
	[ydim2,xdim2] = size( IM.im ); % defaults
	radius2	= min( xdim2, ydim2 ) / 3.2; % defaults

	xdim	= round( str2num( get( H.temp(1), 'String' ) ) );
	if( isempty(xdim) ) xdim = xdim2; end;
	xdim	= max( xdim, 8 );
	xdim	= min( xdim, 1024 );
	set( H.temp(1), 'String', xdim );

	ydim	= round( str2num( get( H.temp(2), 'String' ) ) );
	if( isempty(ydim) ) ydim = ydim2; end;
	ydim	= max( ydim, 8 );
	ydim	= min( ydim, 1024 );
	set( H.temp(2), 'String', ydim );

	radius	= str2num( get( H.temp(4), 'String' ) );
	if( isempty(radius) ) radius = radius2; end;
	radius	= max( radius, 4 );
	radius	= min( radius, floor( min(xdim,ydim)/2 ) );
	set( H.temp(4), 'String', radius );

	vals 	= [xdim ydim radius];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function[ disc ] = make_disc( xdim, ydim, radius )

	xdim		= round( xdim );
	ydim		= round( ydim );
	radius		= radius*cos(atan(max(xdim,ydim)/min(xdim,ydim))); % ?
        vals    	= [1 0];	% values inside and outside of disk
        width   	= 0.025;	% transition width in [0 1]
        origin  	= [xdim/2 ydim/2];     
        [xramp,yramp] 	= meshgrid( [1:xdim]-origin(1), [1:ydim]-origin(2) );
	disc    	= (xramp.^2 + yramp.^2) .^ (1/2);
        disc    	= disc(:);
        mindisc 	= min( disc );
        maxdisc 	= max( disc );
        dim		= 512;          % arbitrary;
        width		= round(width * dim);
        X		= pi * [-width-1:1] / (2*width);
        Y		= vals(1) + (vals(2)-vals(1)) * cos(X).^2;
        d		= round(size(Y,2) / 2);
        c		= round( 2*radius/min(xdim,ydim) * dim );
        Y2		= zeros( 1, dim );
        Y2(1:c-d)	= ones( 1, c-d );
        Y2(c-d:c-d+size(Y,2)-1) = Y;
        step    	= (maxdisc-mindisc) / (size(Y2,2)-1);
        X		= [ mindisc : step : maxdisc ];
        disc		= reshape( interp1(X,Y2,disc), ydim, xdim );