%%%
%%% Q_BRIGHTNESS: ADJUST BRIGHTNESS AND CONTRAST.
%%%
%%%		Adjust brightness and contrast of display.  Brightness
%%%		shifts up or down the range of displayed pixel
%%%		intensities, and contrast shrinks or strethces the
%%%		range. Adjustment is done on the colormap axis:
%%%		get(gca,'Clim');  
%%%
%%% DATE: 	June 3, 1998
%%% BY:   	Hany Farid (farid@mit.edu), Copyright 1998, MIT.
%%%
function [] = Q_brightness( parent, first, mode, ax );

if( first )
	if( Q_active(parent,1) )
		return;
	end

	AX	= get( parent, 'CurrentAxes' );
	IM	= Q_get_iminfo( parent );
	[N,X,S]	= Q_hist( IM );
	Q_window( parent, IM, N, X, S );
	Q_sliders( parent, IM, S );
	Q_callbacks( parent, AX );
else
	IM	= Q_get_iminfo( parent );
	vals	= Q_getvals( parent, mode );
	b	= vals(1);		% brightness
	c	= vals(2);		% contrast

	%%% NEW LIMITS FOR COLORMAP (CLIM)
	mid		= mean(IM.cmap(:,1));
	cmap2		= (c * (IM.cmap-mid)) + mid;
	cmap2		= cmap2 + b/(IM.clim(2)-IM.clim(1));
	m		= cmap2(256) - cmap2(1);
	b		= cmap2(1) - m/255;
	clim2		= 255*[(-b/m) ((1-b)/m)];
	clim2		= (IM.clim(2)-IM.clim(1))/255 * clim2 + IM.clim(1);
	clim2		= sort( clim2 );

	%%% CHECK IF DONE
	if( mode == 3 ) % OK
		IM.clim = clim2;
		Q_resetfig( parent, IM );
		return;
	end

	%%% UPDATE HISTOGRAM AXIS
	xlim(1)	= min( min(IM.im(:)), min(clim2) );
	xlim(1) = xlim(1)-0.05*abs(xlim(1));
	xlim(2)	= max( max(IM.im(:)), max(clim2) );
	xlim(2) = xlim(2)+0.05*abs(xlim(2));
	set( vals(3), 'XLim', xlim );
	set( vals(3), 'XTick', round(clim2*10)/10 );

	%%% UPDATE COLORMAP AXIS
	set( ax, 'CLim', clim2 );
end	

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_window( parent, IM, N, X, S )

	Q_expandfig( parent, 'Brightness/Contrast', 1, 1 );
	Q_show_hist( parent, [0.6 0.55 0.37 0.35], N, X, S );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_sliders( parent, IM, S )

	H 		= Q_get_handles( parent );
	limit		= max( abs([S.minval S.maxval]) );

	%%% BRIGHTNESS SLIDER/EDIT/LABEL
	H.temp(1)	= Q_input( parent, [0.6 0.37 0.25 0.06], ...
				[-limit limit 0 0.01 0.1], 'slider' );
	H.temp(2)	= Q_input( parent, [0.86 0.37 0.12 0.06], 0, 'edit' );
	H.temp(3) 	= Q_input( parent, [0.86 0.30 0.12 0.06], ...
				'Brightness', 'text' );

	%%% CONTRAST SLIDER/EDIT/LABEL
	H.temp(4)	= Q_input( parent, [0.6 0.2 0.25 0.06], ...
				[0.1 10 1 0.001 0.01], 'slider' );
	H.temp(5)	= Q_input( parent, [0.86 0.2 0.12 0.06], 1, 'edit' );
	H.temp(6)	= Q_input( parent, [0.86 0.13 0.12 0.06], ...
				'Contrast', 'text' );

	subplot('position', [0.6 0.55 0.37 0.35]);
	H.temp(7)	= get( parent,'CurrentAxes' );
	Q_set_handles( H, parent );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = Q_callbacks( parent, ax )

	H 	= Q_get_handles( parent );
	cmd1 	= sprintf( 'Q_brightness(%d,0,1,%.20f);', parent, ax );
	cmd2 	= sprintf( 'Q_brightness(%d,0,2,%.20f);', parent, ax );
	cmd3 	= sprintf( 'Q_brightness(%d,0,3,%.20f);', parent, ax );

	%%% BRIGHTNESS SLIDE/EDIT
	set( H.temp(1), 'Callback', cmd1 );
	set( H.temp(2), 'Callback', cmd2 );

	%%% CONTRAST SLIDE/EDIT
	set( H.temp(4), 'Callback', cmd1 );
	set( H.temp(5), 'Callback', cmd2 );

	set( H.ok, 'Callback', cmd3 );

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ vals ] = Q_getvals( parent, mode )

	H		= Q_get_handles( parent );

	if( mode == 1 | mode == 3 | mode == 4 ) % SLIDER INPUT
		b	= round( 100*get( H.temp(1), 'Value' ) ) / 100;
		c	= round( 100*get( H.temp(4), 'Value' ) ) / 100;
		set( H.temp(2), 'String', b );
		set( H.temp(5), 'String', c );
	elseif( mode == 2 ) 			% EDIT INPUT
		b	= str2num( get( H.temp(2), 'String' ) );
		b	= max( b, get( H.temp(1), 'Min' ) );
		b	= min( b, get( H.temp(1), 'Max' ) );
		c	= str2num( get( H.temp(5), 'String' ) );
		c	= max( c, get( H.temp(4), 'Min' ) );
		c	= min( c, get( H.temp(4), 'Max' ) );
		if( isempty(b) ) b = 0;  end;
		if( isempty(c) ) c = 1;  end;
		set( H.temp(1), 'Value', b );
		set( H.temp(2), 'String', b );
		set( H.temp(4), 'Value', c );
		set( H.temp(5), 'String', c );
	end
	vals 	= [b c H.temp(7)]; % bright/contrast/axes handle