function imgIC = vsdIntensityContrastCorrection(img,crop);
% function imgIC = vsdIntensityContrastCorrection(img,crop);
%
% Intensity and contrast normalization. 
%
% Corrects for the intensity gradient applying a wiener-like filtering.
% Then removes mean intensity and normalizes by the contrast (standard
% deviation), both estimated by minimizing a robust error measure between
% the sampled histogram and a mixture of 2 gaussians. 
%
% img: input image (x,y).
%
% crop: optional crop region, 2x2 matrix as follows.
%          [startx starty;
%           endx   endy  ]
%
% djh 7/2008
% Based on mrAlign code originally written by Oscar Nestares and described
% in the Appendix of: Nestares O, Heeger DJ, Robust multiresolution
% alignment of MRI brain volumes. Magn Reson Med 43:705-715, 2000.

imgIC = zeros(size(img));

% Estimate intensity and noise
[int noise] = estIntGrad(img);

% Normalize
sigma2 = mean(noise(:))/2;
imgIC = (img.* int) ./ (int.^2 + noise + sigma2);

% Crop img
if ieNotDefined('crop')
    imgCrop = imgIC;
else
    imgCrop = imgIC([crop(1,1):crop(2,1)],[crop(1,2):crop(2,2)]);
end

% Build the histogram
II = find(~isnan(imgCrop));

% [h x] = regHistogram(imgCrop(II), 256); %commented out by TM
[h x] = hist(imgCrop(II), 256); % regHistogram -> hist, change made by TM

% Normalize the histogram
h = h/(sum(h)*mean(diff(x)));

% Initial parameters: choose the first gaussian with the actual mean and
% variance, and the second gaussian around 1 and with 1/10 of the actual
% variance, both with weights of 0.5 
pinit = [mean(imgCrop(II)) var(imgCrop(II)) 1 var(imgCrop(II))/10 0.5 0.5];

% Minimize robust measure of the error
p = fminsearch('regErrGaussRob', pinit, [], x, h);

% Select the mean closer to 1
if abs(p(1)-1)>abs(p(3)-1)
	mu = p(3); sigma2 = p(4);
else
	mu = p(1); sigma2 = p(2);
end

% Normalize, saturating for low and high values
limit = 4;  % 4*std
imgIC = (imgIC - mu)/sqrt(sigma2);
low = imgIC < -limit;
high = imgIC > limit;
imgIC = imgIC .* ((~low) & (~high)) + (-limit)*low + limit*high;

return

function [intensityGradient, noise] = estIntGrad(inp, lpf);
% estIntGrad - Estimates the intensity gradient, using local mean
%
%    [Int, Noise] = estIntGrad(inp, <lpf>);
%
% Inputs:
%  inp - input image affected by the intensity gradient
%  lpf - low pass filter (applied separably to x,y) used to
%        compute the local mean
%
% Outputs:
%  Int   - Estimated intensity
%  Noise - Estimated power-spatial distribution of the noise, as the
%          local variance
%
% djh 7/2006
% modified from mrAlign code originally written by Oscar Nestares%

% default low-pass filter
if ~exist('lpf')
  lpf = conv([1 4 6 4 1]/16, [1 4 6 4 1]/16);
  %lpf = conv(lpf, lpf);
  %lpf = conv(lpf, lpf);
end

% estimate the intensity as the local mean
intensityGradient = conv2sep(inp, lpf, lpf, 'same');

% estimates the noise as the mean local variance
noise = conv2sep((inp-intensityGradient).^2, lpf, lpf, 'same');

return


%%%%%%%%%%%%%%
% Test/debug %
%%%%%%%%%%%%%%

load images/Image0909.mat
img = double(Im);
crop = [128 128; 384 384];
normImg = vsdIntensityContrastCorrection(img,crop);


