function  [v, w] = renormalize(f, xp, yp, flow)
% Kanatani's renormalization method for 3D motion estimation.
% Refer to "3-D Interpretation of Optical Flow by Renormalization",
% pp. 267--282, v. 11, n. 3, IJCV93.

% v is unit vector of translation direction and w is rotational
% velocity vector.
% f is focal length, and xp and yp are row vector.
% x- and y-component of the velocity is contained in 
% the 1st and 2nd row of flow.

% Compute the N-vector m from image coordinates.

  n = size(xp, 2);
  m = normc([xp; yp; f*ones(1,n)]); % Normalize each column.
     % This is a 3 x N matrix, where each column contains a N-vector.

% Compute the N-velocity m_dot from optical flow (image velocity).

  tmp = 1./sqrt(xp.^2+yp.^2+f^2);  
  tmp1 = (xp.*flow(1,:)+yp.*flow(2,:)).*(tmp.^3);
  m_dot(1,:) = tmp.*flow(1,:) - tmp1.*xp;
  m_dot(2,:) = tmp.*flow(2,:) - tmp1.*yp;
  m_dot(3,:) = -f*tmp1;
     % This is a 3 x N matrix, where each column contains a N-velocity.

% Compute the twisted flow = m cross m_dot.
  m_dot = cross(m, m_dot);
  
% Compute tensor L (symmetric 3 x 3 matrix).
  L = zeros(3);
  for i = 1:3
    for j = i:3
       L(i,j) = sum(m_dot(i,:).*m_dot(j,:));
         % L is symmetric.
    end
  end

% Compute tensor M (strung into a long vector).
  M = [];
  tmp = zeros(3);
  for i = 1:3
      for j = 1:3
          for k = j:3
              tmp(j,k) = sum(m_dot(i,:).*m(j,:).*m(k,:));
          end
      end
      tmp(2,1) = tmp(1,2);
      tmp(3,1) = tmp(1,3);
      tmp(3,2) = tmp(2,3);
     
      M = [M; tmp(:)];
  end

% Compute tensor N, and save them in a matrix for computing inverse of tensor.
% Refer to Appendix D.
 
  N = zeros(6);
  ind = [1 1; 1 2; 2 2; 1 3; 2 3; 3 3];
  for i = 1:6
    for j = 1:(i-1)
        N(i,j) = N(j,i);
    end
    for j = i:6
        u = ind(i,1);
        v = ind(i,2);
        s = ind(j,1);
        t = ind(j,2);
        N(i,j) = sum(m(u,:).*m(v,:).*m(s,:).*m(t,:));
    end
  end

  N(:, [2 4 5]) = 2 * N(:, [2 4 5]);
  N_inv = inv(N);

% Construct matrix A.
  A = zeros(3);    
  e = 3*(ind(:,2)-1) + ind(:,1); 
        % Because the 2D matrix is saved column-wise in the vector.

  for i = 1:3
    for j = i:3
      Mj = M([(j-1)*9+1 : j*9]);       
      Mj = Mj(e);
      B = N_inv * Mj; 
        % Corresponding to elements of B matrix: [B11;B12;B22;B13;B23;B33]
      B = sparse(ind(:,1), ind(:,2), B);
      B(2,1) = B(1,2);
      B(3,1) = B(1,3);
      B(3,2) = B(2,3);
      B = B(:);

      Mi = M([(i-1)*9+1 : i*9]);       

      A(i,j) = sum(Mi.*B); 
         % A is symmetric.
    end
  end
  A = L - A;
  A(2,1) = A(1,2);
  A(3,1) = A(1,3);
  A(3,2) = A(2,3);

% Construct matrix B.
% The moment matrix = m * m'; 
  clear B
  B = n*eye(3) - m * m';

% Solve the generalized eigenvalue problem: A v = c B v, where c is the 
% the smallest eigenvalue to which eigenvector v corresponds. 
% c measures the mean square noise level: c = sigma^2 / 2.
% Refer to Appendix E.

  inv_sqrt_B = inv(sqrtm(B));
  v = normc(inv_sqrt_B * MinEigen(inv_sqrt_B*A*inv_sqrt_B));
                           % Translation direction.

% Compute symmtric matrix K, then rotation velocity w.
  K = zeros(6,1);

  for m = 1:3
      Mm = M([(m-1)*9+1 : m*9]);       
      Mm = Mm(e);
      B = N_inv * Mm; 
        % Corresponding to elements of B matrix: [B11;B12;B22;B13;B23;B33]
      
      K = K - v(m) * B;      
  end

   % K corresponds to elements of K matrix: [K11;K12;K22;K13;K23;K33]
  K = sparse(ind(:,1), ind(:,2), K);
  K(2,1) = K(1,2);
  K(3,1) = K(1,3);
  K(3,2) = K(2,3);

  Kv = K * v;
  w = 0.5*(trace(K)+3*dot(v,Kv))*v - 2*Kv;
















